/*
cdcd - Command Driven CD player
Copyright (C) 1998-99 Tony Arcieri
Copyright (C) 2001, 2002, 2003 Fabrice BAUZAC

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*

  TODO

  In the ff and rew functions (and possibly others), check for the
  track type before playing (test: put the CD "Incubus -
  S.C.I.E.N.C.E.", then play 12, then ff 20:).

  It seems that track 12 of Incubus isn't really 18:34 minutes long
  (try to "play 12 16:2", and "play 12 16:3").

*/

#include <stdio.h>
#include <errno.h>

#ifdef HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#include "cmdline.h"
#include "conf.h"
#include "str.h"
#include "cmd_cdcd.h"
#include "global.h"
#include "rlhist.h"
#include "cmd_access.h"
#include "cmd_sites.h"
#include "interface.h"
#include "cmd_edit.h"

Execfunc cmd_play, cmd_access, cmd_ext, cmd_stop, cmd_close,
  cmd_pause, cmd_resume, cmd_toggle, cmd_rew, cmd_ff, cmd_rndplay, cmd_next,
  cmd_prev, cmd_setvol, cmd_getvol, cmd_status, cmd_info, cmd_tracks,
  cmd_device, cmd_edit, cmd_refresh, cmd_verbose, cmd_list, cmd_slot,
  cmd_quit, cmd_debug, cmd_help, cmd_sites, cmd_open, cmd_excl;

Helpfunc help_sites;

/* A nonzero return value from the cmd_* functions means "quit the
   loop" (only used by the "quit" command for now) */

struct Command cmds[] = {
  {"!", "[COMMANDLINE]", "Execute COMMANDLINE or enter a system shell",
   cmd_excl, NULL, 0},
  ALIAS ("exit", "quit"),
  {
   "debug", "", "Show debugging information", cmd_debug, NULL, 0},
  ALIAS ("?", "help"),
  {
   "play",
   "[starttrackname/track #] [endtrackname/track #] [min:sec]",
   "By default, starts at the beginning of the CD.  "
   "You may specify tracks using either the number of the "
   "track or a section of the track's name.  The "
   "disc will play from where it starts to the end of the "
   "disc or a track you specify.  You may also specify a "
   "position within the first track to start.",
   cmd_play,
   NULL, 1},
  {
   "stop", "", "Stop the CD, if it is playing.",
   cmd_stop, NULL, 1},
  {
   "open", "",
   "Stop the CD, if it is playing, and open the CD tray.",
   cmd_open, NULL, 1},
  ALIAS ("eject", "open"),
  {
   "close", "", "Close the CD tray.", cmd_close, NULL, 1},
  {
   "pause", "", "Pause the CD, if it is playing.",
   cmd_pause, NULL, 1},
  {
   "resume", "",
   "Resume playing the CD, if it is paused.",
   cmd_resume, NULL, 1},
  {
   "toggle", "",
   "Toggle between pause/resume while playing.",
   cmd_toggle, NULL, 1},
  {
   "ff", "[min:sec]",
   "Advance the current track 15 seconds, "
   "or the specified time.", cmd_ff, NULL, 1},
  {
   "rew", "[min:sec]",
   "Go back in the song 15 seconds, "
   "or the specified time.", cmd_rew, NULL, 1},
  {
   "sites", "", "Go in the sites subshell, to edit the CDDB & CDIndex "
   "server list.", cmd_sites, NULL, 0},
  {
   "next", "[min:sec]",
   "Advance to the next song and "
   "optionally at the specified position.", cmd_next, NULL, 1},
  {
   "prev", "[min:sec]",
   "Return to the previous song "
   "optionally at the specified position.", cmd_prev, NULL, 1},
  {
   "getvol", "", "Display the current volume",
   cmd_getvol, NULL, 1},
  {
   "setvol",
   "[VOL] [f=VOL] [b=VOL] [l=VOL] [r=VOL] [fl=VOL] ... [br=VOL]",
   "Set the current volume.  r means right, l means left, f means front, "
   "b means back.  VOL alone sets everything to VOL.  "
   "Valid volumes: 0 - 255.",
   cmd_setvol, NULL, 1},
  {
   "status", "",
   "Display the bare-bones information about "
   "the status of the CD. For more detailed information, "
   "use 'info'.", cmd_status, NULL, 1},
  {
   "rndplay", "",
   "Select a track at random from "
   "the disc and plays it.", cmd_rndplay, NULL, 1},
  {
   "slot", "disc #", "Specify a CD-ROM changer slot to use.",
   cmd_slot, NULL, 1},
  {
   "list", "",
   "Display the entire contents of your CD-ROM changer.",
   cmd_list, NULL, 1},
  {
   "info", "",
   "Display statistics regarding the CD, "
   "such as name, artist, number of tracks, etc.  "
   "When you invoke this command for the first time on a particular CD, "
   "it will block as it attempts to retrieve CD information from "
   "the CDDB.", cmd_info, NULL, 1},
  {
   "tracks", "",
   "Display album, artist, and all track names.",
   cmd_tracks, NULL, 1},
  {
   "edit", "",
   "Go to the edit subshell, to edit the CD name, artist, and track names.",
   cmd_edit, NULL, 0},
  {
   "ext", "[TRACKNAME | TRACK# | \"disc\"]",
   "Display extended information for a track "
   "if it is available, or for the whole disc, "
   "by passing the \"disc\" parameter", cmd_ext, NULL, 1},
  {
   "refresh", "",
   "Update an entry in the local CDDB cache",
   cmd_refresh, NULL, 1},
  {
   "device", "[device name]",
   "Change the default CD-ROM device "
   "(/dev/cdrom).  The new device name is stored in "
   "~/.cdcdrc and will become the default used by cdcd.",
   cmd_device, NULL, 1},
  {
   "verbose", "[on/off]",
   "Toggle whether commands display output when "
   "not absolutely necessary. Use `off' for automated "
   "CDDB operation, or \"on\" for verbose.", cmd_verbose, NULL, 1},
  {
   "quit", "", "Exit cdcd (CD keeps playing).",
   cmd_quit, NULL, 0},
  {
   "access", "",
   "Go in the access subshell, to change the access method to the cddb "
   "databases.", cmd_access, NULL, 0},
  {
   "help", NULL, "Oh please, this isn't Windows.", cmd_help, NULL, 0},
  {
   NULL, NULL, NULL, NULL, NULL, 0}
};

/* Concatenate the string array `S' into a space-separated newly
   malloc'ed string.

   For example: {"hello","world","!"} -> "hello world !".  */
/* To speed this up is possible but pointless */
static char *
concatnew (char **s)
{
  int size;
  {
    char **p;
    /* Begin with size=1, to count the NUL character at the end of the
       string.  */
    for (p = s, size = 1; *p; p++)
      size += strlen (*p) + 1;
    if (p != s)
      --size;			/* Chomp one space if there is >=1 elements. */
  }
  {
    char *ret, *p;
    int first;

    p = ret = (char *) xmalloc (size);
    first = 1;

    /* The invariant property of `p' is that it points to the end of the
       `ret' string; it's where we'll put the NUL character at the
       end.  */
    /* `ret' is a constant and is the address of our string.  */
    for (; *s; s++)
      {
	if (first)
	  first = 0;
	else
	  *p++ = ' ';
	strcpy (p, *s);
	p += strlen (*s);
      }
    *p = 0;			/* Now we put the NUL character (no need to put it >1
				   times).  */
    return ret;
  }
}

int
cmd_toggle (char **argv)
{
  struct disc_info disc;
  NOARG;
  cd_stat (cd_desc, &disc);
  switch (disc.disc_mode)
    {
    case CDAUDIO_PLAYING:
      cd_pause (cd_desc);
      break;
    case CDAUDIO_PAUSED:
      cd_resume (cd_desc);
      break;
    case CDAUDIO_COMPLETED:
    case CDAUDIO_NOSTATUS:
      cd_play (cd_desc, 1);
      break;
    default:
      /* Do nothing.  */
      break;
    }
  return XRET_NULL;
}

int
cmd_excl (char **s)
{
  if (!*++s)
    system ("sh");
  else
    {
      char *cmd;
      cmd = concatnew (s);
      system (cmd);
      xfree (cmd);
    }
  return XRET_NULL;
}

#if 0
int
cmd_version (char **s)
{
  char dispbuffer[4096];
  cd_version (dispbuffer, 4095);
  dispbuffer[4095] = 0;		/* I'm not sure whether this is done in
				   cd_version () or not, so it's done here.  */
  printf ("%s %s\n"
	  "Copyright 2001 Tony Arcieri, Fabrice Bauzac\n", PACKAGE, VERSION);
  pprintf (get_width () - 1,
	   "Distributed under the GNU General Public License.  "
	   "See file COPYING for details.");
  putchar ('\n');
  printf ("Using %s.\n", dispbuffer);
  return 0;
}
#endif

/* This accepts three arguments, each may or may not be provided, in
   any order.  */
int
cmd_play (char **argv)
{
  char **arg;
  struct disc_timeval disc_time;
  int start_provided = 0, end_provided = 0, dt_provided = 0;
  struct disc_info disc;

  /* We try to do the lookup only after we have started to play the CD
     (we are lazy).  If it's not possible (if we detect things that
     look like tracknames in the command-line), then we will be forced
     to lookup before the cd_play_track_pos call.  We will do a lookup
     anyway before the end of the function.  */

  /* Start of lazy stuff.  */
  struct disc_data data;
  int done_lookup = 0;
#define FORCE_LOOKUP				\
    do {					\
      if (! done_lookup) {			\
        lookup_now (cd_desc, &data);		\
        done_lookup = 1;			\
      }						\
    } while (0)
  /* End of lazy stuff.  */

  if (cdcd_cd_stat (cd_desc, &disc) < 0)
    return 0;

  /* TODO: play_track and end_track should both be in the range
     0..(MAX-1), and conversion be processed only when outputting...
     Because the current convention [1..MAX] is disturbing!  */

  for (arg = argv + 1; *arg; arg++)
    {
      if (!start_provided && !read_int (*arg, &play_track))
	start_provided = 1;
      else if (!end_provided && !read_int (*arg, &end_track))
	end_provided = 1;
      else if (!dt_provided && !read_tv (*arg, &disc_time))
	dt_provided = 1;

      else
	{			/* The current arg is certainly a track name.  */
	  FORCE_LOOKUP;
	  if (!start_provided
	      && !read_trackname (*arg, &play_track, &data, &disc))
	    {
	      play_track++;
	      start_provided = 1;
	    }
	  else if (!end_provided
		   && !read_trackname (*arg, &end_track, &data, &disc))
	    {
	      end_track++;
	      end_provided = 1;
	    }
	  else
	    {
	      /* TODO: be more descriptive?  */
	      puts ("unknown track or several matching tracks");
	      return 0;
	    }
	}
    }

  if (!dt_provided)
    disc_time.minutes = disc_time.seconds = disc_time.frames = 0;

  if (!start_provided)
    play_track = ((disc.disc_mode == CDAUDIO_PLAYING) ?
		  disc.disc_current_track : 1);

  if (!end_provided)
    end_track = disc.disc_total_tracks;

  if (end_track < play_track)	/* Then swap start and end.  */
    {
      int t = end_track;
      end_track = play_track;
      play_track = t;
    }

  if (play_track < 1 || end_track > disc.disc_total_tracks)
    {
      printf ("invalid track range\n");
      return 0;
    }

  /* This is useless, but...  Hey, who cares?  */
  if (dt2sec (&disc_time) >
      dt2sec (&disc.disc_track[play_track - 1].track_length))
    {
      disc_time.minutes = disc_time.seconds = disc_time.frames = 0;
      play_track++;
      if (play_track > disc.disc_total_tracks)
	{
	  puts ("end of disc reached");
	  cd_stop (cd_desc);
	  play_track = 1;
	  return 0;
	}
    }

  if (disc.disc_track[play_track - 1].track_type == CDAUDIO_TRACK_DATA)
    {
      puts ("cannot play data track");
      return 0;
    }

  cd_play_track_pos (cd_desc, play_track, end_track,
		     disc_time.minutes * 60 + disc_time.seconds);
  FORCE_LOOKUP;
  return XRET_NULL;
#undef FORCE_LOOKUP
}

int
cmd_ext (char **argv)
{
  struct disc_info disc;
  struct disc_data data;
  int track;
  int paramdisc = 0;
  int track_provided = 0;

  if (cdcd_cd_stat (cd_desc, &disc) < 0)
    return 0;
  lookup_now (cd_desc, &data);

  for (argv++; *argv; ++argv)
    {
      if (!track_provided && !read_trackname (*argv, &track, &data, &disc))
	track_provided = 1;
      else if (!track_provided && !read_int (*argv, &track))
	{
	  track--;
	  track_provided = 1;
	}
      else if (!track_provided && !strcasecmp (*argv, "disc"))
	{
	  paramdisc = 1;
	  track_provided = 1;
	}
      else
	{
	  puts ("Error during argument parsing");
	  return 0;
	}
    }

  if (!track_provided)
    track = disc.disc_current_track - 1;

  /* Now, if !paramdisc, then track must be between 0 and nbtracks-1 */

  if (track < 0 || track >= disc.disc_total_tracks)
    {
      printf ("Track %d out of range\n", track + 1);
      return 0;
    }

  if (paramdisc)
    {
      puts (data.data_title);
      if (*data.data_extended)
	puts (data.data_extended);
    }
  else
    {
#define TE data.data_track[track].track_extended
#define TN data.data_track[track].track_name
      if (*TN && strcasecmp (TE, TN))
	puts (data.data_track[track].track_name);
      if (*TE)
	puts (TE);
#undef TN
#undef TE
    }
  return 0;
}

int
cmd_stop (char **argv)
{
  NOARG;
  cd_stop (cd_desc);
  return XRET_NULL;
}

int
cmd_open (char **argv)
{
  NOARG;
  if (cd_stop (cd_desc) == 0)
    cd_eject (cd_desc);
  return XRET_NULL;
}

int
cmd_close (char **argv)
{
  NOARG;
  cd_close (cd_desc);
  return XRET_NULL;
}

int
cmd_pause (char **argv)
{
  NOARG;
  cd_pause (cd_desc);
  return XRET_NULL;
}

int
cmd_resume (char **argv)
{
  NOARG;
  cd_resume (cd_desc);
  return XRET_NULL;
}

/* Common function for cmd_rew and cmd_ff.  Fills *DT with the time in
   ARGV[1].  returns 0 on success, 1 on error. */
int
one_time_argument (char **argv, struct disc_timeval *dt)
{
  if (!argv[1])
    {
      dt->minutes = dt->frames = 0;
      dt->seconds = 15;
      return 0;
    }
  else if (argv[2])
    {
      puts ("this function takes at most one argument");
      return 1;
    }
  else if (read_tv (argv[1], dt))
    {
      puts ("the argument must be of the form MM:SS (e.g. 2:43)");
      return 1;
    }
  return 0;
}

int
cmd_rew (char **argv)
{
  struct disc_timeval disc_time;
  struct disc_info disc;

  if (one_time_argument (argv, &disc_time))
    return 0;

  printf ("Rewinding %d:%02d.\n", disc_time.minutes, disc_time.seconds);
  if (cd_stat (cd_desc, &disc) != 0)
    return XRET_NULL;

#define MINUS(v)    v = -v
  MINUS (disc_time.minutes);
  MINUS (disc_time.seconds);
#undef MINUS

  /* ???  Please explain me this!  -- fb.  */
  if (end_track < disc.disc_current_track ||
      end_track > disc.disc_total_tracks)
    end_track = disc.disc_total_tracks;

  cd_track_advance (cd_desc, end_track, disc_time);

  return XRET_NULL;
}

int
cmd_ff (char **argv)
{
  struct disc_timeval disc_time;
  struct disc_info disc;

  if (one_time_argument (argv, &disc_time))
    return 0;

  printf ("Fast-forwarding %d:%02d.\n", disc_time.minutes, disc_time.seconds);
  if (cd_stat (cd_desc, &disc) != 0)
    return XRET_NULL;

  if (end_track < disc.disc_current_track ||
      end_track > disc.disc_total_tracks)
    end_track = disc.disc_total_tracks;

  cd_track_advance (cd_desc, end_track, disc_time);

  return XRET_NULL;
}

int
cmd_rndplay (char **argv)
{
  struct disc_info disc;
  NOARG;
  if (cd_stat (cd_desc, &disc) != 0)
    return XRET_NULL;

  play_track = (rand () % disc.disc_total_tracks) + 1;

  /* Shouldn't this be end_track = play_track?  -- fb.  */
  end_track = disc.disc_total_tracks;

  cd_play_track (cd_desc, play_track, end_track);

  return XRET_NULL;
}

int
cmd_next (char **argv)
{
  struct disc_timeval disc_time;
  struct disc_info disc;
  int dt_prov = 0;

  disc_time.frames = 0;

  for (++argv; *argv; ++argv)
    {
      if (!dt_prov && !read_tv (*argv, &disc_time))
	dt_prov = 1;
      else
	{
	  puts ("Error during argument parsing");
	  return 0;
	}
    }

  if (!dt_prov)
    disc_time.minutes = disc_time.seconds = disc_time.frames = 0;

  if (cd_stat (cd_desc, &disc) != 0)
    return XRET_NULL;

  if (end_track < disc.disc_current_track
      || end_track > disc.disc_total_tracks)
    end_track = disc.disc_total_tracks;

  if (disc.disc_current_track + 1 > end_track)
    {
      cd_stop (cd_desc);
      return XRET_NULL;
    }
  else
    {
      cd_play_track_pos (cd_desc, disc.disc_current_track + 1,
			 end_track, disc_time.minutes * 60 +
			 disc_time.seconds);
      return XRET_NULL;
    }
}

int
cmd_prev (char **argv)
{
  struct disc_timeval disc_time;
  struct disc_info disc;
  int dt_prov = 0;

  disc_time.frames = 0;

  for (argv++; *argv; ++argv)
    {
      if (!dt_prov && !read_tv (*argv, &disc_time))
	dt_prov = 1;
      else
	{
	  puts ("Error during argument parsing");
	  return 0;
	}
    }

  if (!dt_prov)
    disc_time.minutes = disc_time.seconds = disc_time.frames = 0;

  if (cd_stat (cd_desc, &disc) != 0)
    return XRET_NULL;
  if (end_track < disc.disc_current_track
      || end_track > disc.disc_total_tracks)
    end_track = disc.disc_total_tracks;

  if (disc.disc_current_track - 1 < 1)
    cd_play_track (cd_desc, 1, end_track);
  else
    cd_play_track_pos (cd_desc, disc.disc_current_track - 1,
		       end_track, disc_time.minutes * 60 + disc_time.seconds);
  return XRET_NULL;
}

int
cmd_setvol (char **argv)
{
  struct disc_volume vol;

  /* Flags that mean: "provided or not?"  */
  int fr = 0, fl = 0, br = 0, bl = 0;

#define FL vol.vol_front.left
#define FR vol.vol_front.right
#define BL vol.vol_back.left
#define BR vol.vol_back.right

  cd_get_volume (cd_desc, &vol);

#define CMP(a, b) (!strncmp (*argv, #a #b "=", 3) \
                   || !strncmp (*argv, #b #a "=", 3))

  for (argv++; *argv; argv++)
    {
      if (!fr && CMP (f, r) && !read_int (*argv + 3, &FR))
	fr = 1;
      else if (!fl && CMP (f, l) && !read_int (*argv + 3, &FL))
	fl = 1;
      else if (!br && CMP (b, r) && !read_int (*argv + 3, &BR))
	br = 1;
      else if (!bl && CMP (b, l) && !read_int (*argv + 3, &BL))
	bl = 1;
      else if (!fl && !fr && !strncmp (*argv, "f=", 2)
	       && !read_int (*argv + 2, &FL))
	{
	  FR = FL;
	  fr = fl = 1;
	}
      else if (!bl && !br && !strncmp (*argv, "b=", 2)
	       && !read_int (*argv + 2, &BL))
	{
	  BR = BL;
	  br = bl = 1;
	}
      else if (!fl && !bl && !strncmp (*argv, "l=", 2)
	       && !read_int (*argv + 2, &FL))
	{
	  BL = FL;
	  fl = bl = 1;
	}
      else if (!fr && !br && !strncmp (*argv, "r=", 2)
	       && !read_int (*argv + 2, &FR))
	{
	  BR = FR;
	  fr = br = 1;
	}
      else if (!fr && !fl && !br && !bl && !read_int (*argv, &FL))
	{
	  BR = BL = FR = FL;
	  fr = br = fl = bl = 1;
	}
      else
	{
	  puts ("Argument error");
	  return 0;
	}
    }

#undef CMP
#undef FR
#undef FL
#undef BR
#undef BL

  if (br + bl + fr + fl == 0)	/* no argument */
    {
      puts ("Missing argument");
      return 0;
    }

  if (cd_set_volume (cd_desc, vol) < 0)
    puts ("Invalid volume");

  return 0;
}

int
cmd_getvol (char **argv)
{
  struct disc_volume vol;
  NOARG;
  cd_get_volume (cd_desc, &vol);
  printf ("%-6s%7s%7s\n", "", "Left", "Right");
  printf ("%-6s%7d%7d\n", "Front", vol.vol_front.left, vol.vol_front.right);
  printf ("%-6s%7d%7d\n", "Back", vol.vol_back.left, vol.vol_back.right);
  return 0;
}

int
cmd_status (char **argv)
{
  struct disc_info disc;
  NOARG;
  cd_stat (cd_desc, &disc);
  switch (disc.disc_mode)
    {
    case CDAUDIO_PLAYING:
    case CDAUDIO_PAUSED:
      printf ("%s n%d %02d:%02d.%02d/track %02d:%02d.%02d/disc\n",
	      (disc.disc_mode == CDAUDIO_PLAYING) ? "Playing" : "Paused",
	      disc.disc_current_track,
	      disc.disc_track_time.minutes,
	      disc.disc_track_time.seconds,
	      disc.disc_track_time.frames,
	      disc.disc_time.minutes,
	      disc.disc_time.seconds, disc.disc_time.frames);
      break;
    case CDAUDIO_COMPLETED:
      puts ("Stopped");
      break;
    case CDAUDIO_NOSTATUS:
      puts ("Stopped");
      break;
    default:
      break;
    }
  return 0;
}

void
nameartist (struct disc_data *dd)
{
  printf ("%-15s %s\n", "Album name:", dd->data_title);
  if (!strcasestr (dd->data_artist, "various")
      && strlen (dd->data_artist) > 0)
    printf ("%-15s %s\n", "Album artist:", dd->data_artist);
}

int
cmd_info (char **argv)
{
  struct disc_info disc;
  struct disc_data data;

  NOARG;
  if (cdcd_cd_stat (cd_desc, &disc) < 0)
    return 0;
  lookup_now (cd_desc, &data);

  nameartist (&data);

#define CURDATA data.data_track[disc.disc_current_track - 1]
#define CURDISC disc.disc_track[disc.disc_current_track - 1]
#define ENDDATA data.data_track[end_track - 1]

#define DISC_OF printf ("%02d:%02d of %02d:%02d",  \
                        disc.disc_time.minutes,    \
		        disc.disc_time.seconds,    \
		        disc.disc_length.minutes,  \
		        disc.disc_length.seconds)
#define TRACK_OF printf ("%02d:%02d of %02d:%02d",                   \
			 disc.disc_track_time.minutes,               \
			 disc.disc_track_time.seconds,               \
			 CURDISC.track_length.minutes,               \
			 CURDISC.track_length.seconds)

  switch (disc.disc_mode)
    {
    case CDAUDIO_PLAYING:
      printf ("%-15s %-7d %-15s ",
	      "Total tracks:", disc.disc_total_tracks, "Disc playing:");
      DISC_OF;
      putchar ('\n');
      if (strlen (CURDATA.track_artist) > 0)
	{
	  printf ("%-15s %s / %s %02d ",
		  "Playing:",
		  CURDATA.track_artist,
		  CURDATA.track_name, disc.disc_current_track);
	  TRACK_OF;
	  putchar ('\n');
	}
      else
	{
	  printf ("%-15s %s %02d ",
		  "Playing:", CURDATA.track_name, disc.disc_current_track);
	  TRACK_OF;
	  putchar ('\n');
	}
      if (end_track != disc.disc_total_tracks && end_track != 0)
	{
	  if (strlen (ENDDATA.track_artist) > 0)
	    printf ("%-15s %s / %s %02d\n",
		    "End track:",
		    ENDDATA.track_artist, ENDDATA.track_name, end_track);
	  else
	    printf ("%-15s %s %02d\n",
		    "End track:", ENDDATA.track_name, end_track);
	}
      break;
    case CDAUDIO_PAUSED:
      printf ("%-15s %-7d %-15s ",
	      "Total tracks:", disc.disc_total_tracks, "Disc paused:");
      DISC_OF;
      putchar ('\n');
      printf ("%-15s %s %02d ",
	      "Paused:", CURDATA.track_name, disc.disc_current_track);
      TRACK_OF;
      putchar ('\n');
      if (end_track != disc.disc_total_tracks && end_track != 0)
	printf ("%-15s %s %02d\n",
		"End track:", ENDDATA.track_name, end_track);
      break;
    case CDAUDIO_COMPLETED:
    case CDAUDIO_NOSTATUS:
      printf ("%-15s %-7d %-15s %02d:%02d\n",
	      "Total tracks:",
	      disc.disc_total_tracks,
	      "Disc length:",
	      disc.disc_length.minutes, disc.disc_length.seconds);
      puts ("Stopped");
      break;
    default:
      break;
    }
  return 0;
#undef DISC_OF
#undef TRACK_OF
#undef CURDATA
#undef CURDISC
#undef ENDDATA
}

int
cmd_tracks (char **argv)
{
  int track;
  struct disc_data data;
  struct disc_info disc;

  NOARG;

  if (cdcd_cd_stat (cd_desc, &disc) < 0)
    return 0;

  lookup_now (cd_desc, &data);

  nameartist (&data);

  printf ("%-15s %-7d %-15s %02d:%02d\n\n",
	  "Total tracks:",
	  disc.disc_total_tracks,
	  "Disc length:", disc.disc_length.minutes, disc.disc_length.seconds);

  printf ("%-7s %-11s %s\n", "Track", "Length", "Title");
  {
    int i, l;
    l = get_width () - 1;
    for (i = 0; i < l; ++i)
      putchar ('-');
    putchar ('\n');
  }

#define CURDISC disc.disc_track[track]
#define CURDATA data.data_track[track]

  for (track = 0; track < disc.disc_total_tracks; track++)
    {
      printf ("%2d:%4s [%2d:%02d.%02d]%1s ", track + 1,
	      track == disc.disc_current_track - 1 ? ">" : "",
	      CURDISC.track_length.minutes, CURDISC.track_length.seconds,
	      CURDISC.track_length.frames, "");
      if (*CURDATA.track_artist)
	printf ("%s / ", CURDATA.track_artist);
      if (*CURDATA.track_name)
	printf ("%s ", CURDATA.track_name);
      if (CURDISC.track_type == CDAUDIO_TRACK_DATA)
	fputs ("(data)", stdout);
      putchar ('\n');
    }
  return 0;
#undef CURDISC
#undef CURDATA
}

int
cmd_device (char **argv)
{
  struct cdcdrc cdcdrc_data;

  if (!argv[1])
    {
      printf ("Current device: %s\n", device);
      return 0;
    }
  else if (argv[2])
    {
      puts ("Too many arguments");
      return 0;
    }

  cd_finish (cd_desc);

  if ((cd_desc = cd_init_device (argv[1])) < 0)	/* First attempt.  */
    {
      static const char *fmt = "Error: %s appears to be already mounted.\n";

      if (errno == EBUSY)
	printf (fmt, argv[1]);
      else
	perror (argv[1]);

      if ((cd_desc = cd_init_device (device)) < 0)	/* Second attempt.  */
	{
	  if (errno == EBUSY)
	    printf (fmt, device);
	  else
	    perror (device);
	}
    }
  else
    {
      strncpy (device, argv[1], CDCDRC_DEVICE_LEN - 1);
      device[CDCDRC_DEVICE_LEN - 1] = 0;
      cdcdrc_data.verbosity = verbosity;
      strncpy (cdcdrc_data.device, device, CDCDRC_DEVICE_LEN - 1);
      cdcdrc_data.device[CDCDRC_DEVICE_LEN - 1] = 0;
      cdcdrc_write (&cdcdrc_data);
    }
  return 0;
}

int
cmd_edit (char **argv)
{
  NOARG;
  return cmd_edit_mainloop ();
}

int
cmd_refresh (char **argv)
{
  struct disc_info disc;
  struct disc_data data;
  struct cddb_entry entry;

  NOARG;

  if (cdcd_cd_stat (cd_desc, &disc) < 0)
    {
      puts ("Couldn't stat the CD");
      return 0;
    }

  cddb_stat_disc_data (cd_desc, &entry);

  if (entry.entry_present)
    {
      lookup_now (cd_desc, &data);
      cddb_erase_entry (data);
    }

  lookup_now (cd_desc, &data);
  return 0;
}

int
cmd_verbose (char **argv)
{
  struct cdcdrc cdcdrc_data;

  if (!argv[1])
    {
      printf ("Verbosity is %s.\n", verbosity ? "on" : "off");
      return 0;
    }
  else if (argv[2])
    {
      puts ("Too many arguments");
      return 0;
    }

  if (!strcasecmp (argv[1], "on"))
    verbosity = 1;
  else if (!strcasecmp (argv[1], "off"))
    verbosity = 0;
  else
    {
      puts ("Wrong argument");
      return 0;
    }

  cdcdrc_data.verbosity = verbosity;
  strncpy (cdcdrc_data.device, device, CDCDRC_DEVICE_LEN - 1);
  cdcdrc_data.device[CDCDRC_DEVICE_LEN - 1] = 0;
  cdcdrc_write (&cdcdrc_data);
  return 0;
}

int
cmd_list (char **argv)
{
  struct disc_changer changer;
  int disp_disc;
  NOARG;
  cd_changer_stat (cd_desc, &changer);
#define INFO changer.changer_disc[disp_disc]
  for (disp_disc = 0; disp_disc < changer.changer_slots; disp_disc++)
    if (INFO.disc_present)
      printf ("Disc %2d:%7s %2d:%02d %s\n",
	      disp_disc + 1,
	      "",
	      INFO.disc_length.minutes,
	      INFO.disc_length.seconds, INFO.disc_info);
  return 0;
#undef INFO
}

int
cmd_slot (char **argv)
{
  int cur_disc;

  if (!argv[1] || argv[2])
    {
      puts ("wrong number of arguments");
      return 0;
    }
  else if (read_int (argv[1], &cur_disc))
    {
      puts ("wrong argument");
      return 0;
    }
  else if (--cur_disc < 0)	/* TODO: What's the other boundary? */
    {
      printf ("Invalid slot %d\n", cur_disc + 1);
      return 0;
    }

  cd_changer_select_disc (cd_desc, cur_disc);
  return 0;
}

int
cmd_quit (char **argv)
{
  NOARG;
  return XRET_QUIT;
}

int
cmd_debug (char **argv)
{
#define CDJ CMD_DEBUG_JUSTIFY	/* Temporary nickname.  */
  struct disc_info disc;
  struct disc_data data;
  int track;

  if (cdcd_cd_stat (cd_desc, &disc) < 0)
    return 0;

  lookup_now (cd_desc, &data);

  puts ("Debugging information:");
  printf ("%" CDJ "15s %s\n", "CDI id:", data.data_cdindex_id);
  printf ("%" CDJ "15s %08lx\n", "CDDB id:", data.data_id);
  printf ("%" CDJ "15s %s\n", "Disc genre:", cddb_genre (data.data_genre));
  printf ("%" CDJ "15s ", "Disc mode:");
  switch (disc.disc_mode)
    {
    case CDAUDIO_PLAYING:
      puts ("Playing");
      break;
    case CDAUDIO_PAUSED:
      puts ("Paused");
      break;
    case CDAUDIO_COMPLETED:
      puts ("Stopped");
      break;
    case CDAUDIO_NOSTATUS:
      puts ("Stopped");
      break;
    default:
      break;
    }
  printf ("%" CDJ "15s %02d:%02d\n", "Track time:",
	  disc.disc_track_time.minutes, disc.disc_track_time.seconds);
  printf ("%" CDJ "15s %02d:%02d\n", "Disc time:",
	  disc.disc_time.minutes, disc.disc_time.seconds);
  printf ("%" CDJ "15s %02d:%02d\n", "Disc length:",
	  disc.disc_length.minutes, disc.disc_length.seconds);
  printf ("%" CDJ "15s %d\n", "Disc track:", disc.disc_current_track);
  printf ("%" CDJ "15s %d\n", "Total tracks:", disc.disc_total_tracks);
  for (track = 0; track < disc.disc_total_tracks; track++)
    {
      printf ("Track %-2d %" CDJ "6s %d\n", track + 1, "LBA:",
	      disc.disc_track[track].track_lba);
      printf ("%" CDJ "15s %02d:%02d\n", "length:",
	      disc.disc_track[track].track_length.minutes,
	      disc.disc_track[track].track_length.seconds);
    }
  return 0;
#undef CDJ
}

int
cmd_help (char **argv)
{
  cmdhelp (cmds, argv, 0);
  return 0;
}

int
cmd_access (char **argv)
{
  NOARG;
  return cmd_access_mainloop ();
}

int
cmd_sites (char **argv)
{
  NOARG;
  return cmd_sites_mainloop ();
}

#if 0
char *
cdcd_command_matcher (char *text, int state)
{
  static int list_index, len;

  if (!*text)
    return NULL;

  if (!state)
    {
      list_index = 0;
      len = strlen (text);
    }

  for (; cmds[list_index].name; ++list_index)
    if (!strncmp (cmds[list_index].name, text, len))
      return strdup (cmds[list_index++].name);

  return NULL;
}
#endif

void
init_cmd_cdcd ()
{
  sort_commands (cmds);
}

int
cmd_cdcd_execute_commandline (char *s)
{
  char **p;
  int r;
  p = my_tokenize (s);
  r = execute_command (p, cmds);
  freev0 (p);
  return r;
}

char *
cdcd_command_matcher (const char *text, int state)
{
  return command_matcher (cmds, text, state);
}

void
cmd_cdcd_mainloop ()
{
  rl_attempted_completion_function = (CPPFunction *) cdcd_completion;
  cmd_mainloop (&cdcd_command_matcher, "cdcd> ", cmds);
}

int
cmd_cdcd_execute (char **argv)
{
  return execute_command (argv, cmds);
}
