-- -*- coding: utf-8 -*-
--------------------------------------------------------------------------------
-- Copyright 2024  Nathan Grieve
--
-- This program is free software: you can redistribute it and/or modify it under
-- the terms of the GNU General Public License as published by the Free Software
-- Foundation, either version 3 of the License, or (at your option) any later
-- version.
--
-- This program is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
-- FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
-- details.
--
-- You should have received a copy of the GNU General Public License along with
-- this program.  If not, see <http://www.gnu.org/licenses/>.
--------------------------------------------------------------------------------


newPackage(
    "AbstractSimplicialComplexes",
    Version => "0.1",
    Date => "30 September 2024",
    Headline => "Abstract Simplicial Complexes",
    Authors => {{ Name => "Nathan Grieve", Email => "nathan.m.grieve@gmail.com", HomePage => "https://sites.google.com/view/nathan-grieve"}},
    AuxiliaryFiles => false,
    DebuggingMode => false,
    PackageImports => {"Complexes"},
    PackageExports => {"Complexes"},
    Keywords => {"Combinatorial Commutative Algebra"}
    )

export {"AbstractSimplicialComplex", "abstractSimplicialComplex","simplicialChainComplex", "reducedSimplicialChainComplex", "ambientAbstractSimplicialComplexSize",
    "ambientAbstractSimplicialComplex", "abstractSimplicialComplexFacets", "randomAbstractSimplicialComplex", "randomSubSimplicialComplex",
     "inducedSimplicialChainComplexMap","inducedReducedSimplicialChainComplexMap" 
    }

-* Code section *-

---------------------------------------
-- spots
----------------------------------------
-- the spots method is extremely useful
-- but we don't export it 
-----------------------------------------

spots = method()

spots Complex := List => (
  C -> (c := concentration C; toList(c_0 .. c_1)))

max Complex := K -> max spots K
min Complex := K -> min spots K


---------------------------------------
--------------------------------------

--------------------------
-- Abstract Simplicial Complex 
-------------------------

-- The idea is to make an Abstract Simplicial Complex as a Type of HashTable as a means
-- For working with Abstract Simplicial Complexes ---
-- The integer keys will output the list of i-simplicies

AbstractSimplicialComplex = new Type of HashTable
AbstractSimplicialComplex.synonym = "abstract simplicial complex"

AbstractSimplicialComplex.GlobalAssignHook = globalAssignFunction
AbstractSimplicialComplex.GlobalReleaseHook = globalReleaseFunction
describe AbstractSimplicialComplex := K -> net expression K


new AbstractSimplicialComplex := AbstractSimplicialComplex =>(cl) -> (
    K := newClass(AbstractSimplicialComplex, new HashTable); -- sigh
    K)

spots AbstractSimplicialComplex := List => (
  K -> sort select(keys K, i -> class i === ZZ))

-- return the p-faces of a simplicial complex

AbstractSimplicialComplex _ ZZ := AbstractSimplicialComplex => (K,p) -> (
  if K#?p then K#p 
  )

-- given a list of subsets L and A \in L decide if A is maximal

isMaximal :=(x,L) -> (
myList := select(L,i -> isSubset(x,i));
if #myList == 1 then
return true
else return false
    )

-- select the maximal subsets (i.e., facets) of a list of subsets

listFacets := (L) -> (
select(L,i-> isMaximal(i,L))
    )


--- return the facets of a simplicial complex

abstractSimplicialComplexFacets = method()

abstractSimplicialComplexFacets(AbstractSimplicialComplex) := List => K ->(
    L := flatten(apply(spots K, i-> K_i));
    return listFacets(L)
    )


--- decide if two simplicial complexes are equal

AbstractSimplicialComplex == AbstractSimplicialComplex := Boolean => (K,L) ->(
    return (abstractSimplicialComplexFacets K) == (abstractSimplicialComplexFacets L)
    )


--- returns the dimension of a simplicial complex


dim AbstractSimplicialComplex := ZZ => (K) -> (
    return (max apply(abstractSimplicialComplexFacets(K), i -> #i) - 1)
    )

--- Constructors for AbstractSimplicialComplexs

abstractSimplicialComplex = method()

-- the most basic constructor of a AbstractSimplicialComplex

-- The idea is to make a simplical complex starting from a list of faces.
--  The list of faces need not be facets.
--  The constructor returns the simplicial complex (with all of its faces) that is
--  generated by this list of faces 
--  By default, it is assumed that the kfaces are all lex ordered positive integers 

makeKFaces := (L,k) -> (
    toList(set(flatten(apply(#L, i -> subsets(sort L_i,k)))))
    )

makeAllFaces := (L) -> (
    numberOfFaces := #L;
--  find the highest dimensional face    
    n := max(apply(numberOfFaces, i-> # (L_i)));
    flatten(for k from 0 to n list {k-1 => sort makeKFaces(L,k)})
    )


abstractSimplicialComplex(List) := AbstractSimplicialComplex => L -> (
    return new AbstractSimplicialComplex from makeAllFaces(L)
    )


--- The following method will make the (n-1)-dimensional n-simplex on [n] = {1,...,n}
abstractSimplicialComplex(ZZ) := AbstractSimplicialComplex => (n) -> (
    L := for i from 1 to n list i;
    return abstractSimplicialComplex({L})
    )

--- Make the "r-skeleton" on [n] = {1,...n}

abstractSimplicialComplex(ZZ,ZZ) := AbstractSimplicialComplex => (n,r) -> (
    return abstractSimplicialComplex subsets(for i from 1 to n list i,r)
    )

--

-- making random simplicial complexes --

-- make a random subset of {1,...,n}

randomSubset = method()

randomSubset(ZZ) := List => (n) -> (
   setRandomSeed(currentTime());
   k := random(1,n); 
   sort unique (for i from 1 to k list (random(1,n)))
    )

-- random size r subset --

randomSubset(ZZ,ZZ) := List => (n,r) -> (
   setRandomSeed(currentTime());
   sort unique (for i from 1 to r list (random(1,n)))
    )


-- make a random subset of a given set

randomSubset(List) := List => (L) -> (
    setRandomSeed(currentTime());
    n := #L;
    k := random(0,n);
    mySubset := subsets(L,k);
    mySubset_(random(binomial(n,k)))
    )

-- a variant of this is would be to make a random k element subset of a given set --

-- The following will make a "random" simplicial complex on {1,...,n} --

randomAbstractSimplicialComplex = method()

randomAbstractSimplicialComplex(ZZ) := AbstractSimplicialComplex => (n) -> (
     setRandomSeed(currentTime());
     listLength := 1 + random(2^n);
     abstractSimplicialComplex unique(for i from 1 to listLength list randomSubset(n))
     )

------

--  It also seems like a good idea to make a random simplicial complex
--  on [n] with dimension at most equal to r

-----

randomAbstractSimplicialComplex(ZZ,ZZ) := AbstractSimplicialComplex =>(n,r) -> (
     setRandomSeed(currentTime());
     listLength := 1 + random(binomial(n,r));
     abstractSimplicialComplex unique(for i from 1 to listLength list randomSubset(n,r))
    )


-- can we make the random complex Y_d(n,m) which has vertex set
-- [n] and complete (d − 1)-skeleton, and has exactly m d-dimensional faces,
-- chosen at random from all binomial(binomial(n,d+1),m) possibilities?
-- Such random complexes appear in lots of different contexts including in the article
-- COHEN–LENSTRA HEURISTICS FOR TORSION IN HOMOLOGY OF RANDOM COMPLEXES
-- (MATTHEW KAHLE, FRANK H. LUTZ, ANDREW NEWMAN, AND KYLE PARSONS) --

randomAbstractSimplicialComplex(ZZ,ZZ,ZZ) := (n,m,d) -> (
    setRandomSeed(currentTime());
    L := for i from 1 to n list i;
    dDimlSubsets := subsets(L,d+1);
    randomFaces := for i from 1 to m list (dDimlSubsets#(random(binomial(n,d+1))));
    append(append(randomFaces,{L}),subsets(L,d));
    return abstractSimplicialComplex(randomFaces)
    )

randomSubSimplicialComplex = method()

randomSubSimplicialComplex(AbstractSimplicialComplex) := AbstractSimplicialComplex => (K) -> (
 setRandomSeed(currentTime());
 L := abstractSimplicialComplexFacets K;
 abstractSimplicialComplex unique apply(L, i-> randomSubset(i))
)

---

-- ambient simplicial complex

ambientAbstractSimplicialComplexSize = method() -- return the size of the underlying ambient simplex

ambientAbstractSimplicialComplexSize(AbstractSimplicialComplex) := (K) -> (
    max flatten(K_0)
    )


ambientAbstractSimplicialComplex = method() -- return the underlying ambient simplex 


ambientAbstractSimplicialComplex(AbstractSimplicialComplex) := AbstractSimplicialComplex => (K) -> (
    return abstractSimplicialComplex(ambientAbstractSimplicialComplexSize(K))
    )

---------------------------
--- boundary maps ---
------------------------
  
--  There are many ways to approach 
--  The simplical boundary map
-- For X a given simplicial complex
--  Perhaps the most straight forward way
--  is via 
-- \partial_k : C_k(X) \rightarrow C_{k-1}(X)
--  Here C_k(X) is the free 
-- \ZZ-module (or \kk-vector space)
-- on the set of k+1 simplicies 
-- (i.e., the set of k+1 combinations
-- of {1,...,n})

--  Given input a k+1 lex segment a = [a_0,...,a_k] ---- i.e., a k-face
--  Compute its image under the boundary map
--  It seems most straight forward to give the 
-- output as a k+1 tuple with entry i having the
-- form [(-1)^i,d_i(a)]
-- here d_i(a) is a with the i-th lex entry removed
--  the following is more simply just for
-- testing purposes and is not used explicitly in the sequel
--partial := (L) -> ( 
 -- apply(0 .. (#L-1), i -> {(-1)^i, drop(L,{i,i})})

 
-- The following function seems useful to  
-- useful to construct
-- the simplicial chain complex map
-- given a k-face y and a k+1 - face x
-- decide if it equals \partial(x,i)
-- for some i


isDLexSeqI := (y,x) -> (
    k := #y;
    sign := 0;
    for i from 0 to # x do (
        z := drop(x,{i,i});
        if y == z then (sign = (-1)^i;
            break);
	);
    return sign
)


-- make a constructor for making matrices
-- that represented the simplicial boundary
-- maps of a given simplical complex
-- what follows appears to work OK which is good!

simplicialMakeMatrix = method()

simplicialMakeMatrix(List,List) := (kPlusOneFaces,kFaces) -> (
    n := # kPlusOneFaces;
    m := # kFaces;
    matrixList := 
    	for i from 0 to m-1 list (
    for j from 0 to n-1 list (
	   isDLexSeqI((kFaces)#i,(kPlusOneFaces)#j))
	);
    return matrix(matrixList)
)


--  We can finally make the entire reduced homology chain complex in the following way
-- Given as input the simplcial complex represented as a simplicial set --
--  This will produce the reduced chain complex (so the empty set will
--  appear in the chain complex)

reducedSimplicialChainComplex = method() -- return the chain complex (with contribution from the empty face) that is associated to a simplicial set (i.e., an abstract simplicial complex)

reducedSimplicialChainComplex(AbstractSimplicialComplex) := Complex => (L) ->
(
    n := max spots L;
    if n == -1 then (return complex hashTable {-1 => map(ZZ^0,ZZ^1,zero)})
    else(
    mapsList := for i from 0 to n list (i => simplicialMakeMatrix(L#i,L#(i-1)));
    append(mapsList,-1 => map(ZZ^0,target(mapsList#0)#1,zero)););
    return complex hashTable mapsList
	)

simplicialChainComplex = method() --  return the non-reduced simplicial chain complex (i.e., the chain complex with no contribution from the empty face)

simplicialChainComplex(AbstractSimplicialComplex) := Complex => (L) ->
(
 return(naiveTruncation(reducedSimplicialChainComplex L, 0, infinity))
	)
        
---  Another method that is of interest,
--  is to give an inclusion (or more general a morphism)
---- of simplicial complexes and then compute
--  the induced chain complex morphism of SimplicialChainComplexes
--- An important special case is to view a
--  sub simplicial complex of the full simplicial complex (simplex) and then to compute
--- the corresponding induced inclusion morphism.

---  A first step is to make an k-face inclusion map given an inclusion of simplicial sets 
---  Assume that L <= H
---  If L_k has no faces then the method returns an error message
---  Otherwise the method produces the appropriate matrix
---  That induces the corresponding inclusion map


inducedKFaceSimplicialChainComplexMap = method()

inducedKFaceSimplicialChainComplexMap(ZZ,AbstractSimplicialComplex,AbstractSimplicialComplex) := (k,H,L) ->
(
     M := L_k;
     N := H_k;
     n := # M;
     m := # N;
     myMatrixList := for i from 0 to m-1 list (
	 for j from 0 to n-1 list (
	     if N#i == M#j then 1 else 0 
	 )
     );
     return matrix myMatrixList
)

--If H <= L then give the induced chain complex map for (non-reduced) simplicalChainComplexes

inducedSimplicialChainComplexMap = method()

inducedSimplicialChainComplexMap(AbstractSimplicialComplex,AbstractSimplicialComplex) := (L,H) ->
(
    h := simplicialChainComplex H;
    l := simplicialChainComplex L;
    if ((abstractSimplicialComplex {{}}) == H) then return map(l,h,zero)
    else( 
    f := hashTable apply(spots h, i -> if i == -1 then i => map(l_(-1),h_(-1),zero) else i => inducedKFaceSimplicialChainComplexMap(i,L,H));
    return map(l,h,f);
    )
)

--If H <= L then give the induced chain complex map for reduced simplicalChainComplexes

inducedReducedSimplicialChainComplexMap = method()

inducedReducedSimplicialChainComplexMap(AbstractSimplicialComplex,AbstractSimplicialComplex) := (L,H) -> (
    h := reducedSimplicialChainComplex H;
    l := reducedSimplicialChainComplex L;
    if ((abstractSimplicialComplex {{}}) == H) then return map(l,h, hashTable {-2 => map(l_(-2),h_(-2),zero), -1 => map(l_(-1),h_(-1),id_(h_(-1)))})
    else( 
    f := hashTable apply(spots h, i -> if i == -1 then i => map(l_(-1),h_(-1),id_(h_(-1))) else i => inducedKFaceSimplicialChainComplexMap(i,L,H));
    return map(l,h,f);
    )
)

-----
 
-* Documentation section *-
beginDocumentation()

document { 
  Key => AbstractSimplicialComplexes,
  Headline => "a package for working with abstract simplicial complexes",
  "In this package our conventions are that `abstract simplicial complexes' have vertices supported on the set [n] := {1,...,n}.
  Our aim is to provide a methology for working with such objects directly.  
  In this regard our approach differs from that of the package 
  SimplicialComplexes.m2.
  Here, we are especially interested in homological aspects of SimplicialComplexes and our approach is to implement such simplicial complexes as certain graded lists. 
  In particular, we provide methods for working with the chain complexes that are associated to each abstract simplicial complex.  
  We also give some functionality for producing random simplicial complexes.",

   SUBSECTION "An overview of this package",
   UL {
    TO "How to make abstract simplicial complexes", 
    TO "How to make reduced and non-reduced simplicial chain complexes",
    TO "Calculations with random simplicial complexes"
    },
}


--------------------------------------------
-- Package overview examples ---------------
--------------------------------------------

doc ///
     Key
     	  "How to make abstract simplicial complexes"
     Headline
     	  Using the type AbstractSimplicialComplexs to represent abstract simplicial complexes
     Description
     	  Text	  
	     The type AbstractSimplicialComplex is a data type for working with
	     abstract simplicial complexes with vertices supported on [n] = {1,...,n}.
	     Here we illustrate some of the most basic ways to interact with this data type.
          Text
	     The simplicial complex that is generated by {1,2,3,4}, {2,3,5} and {1,5} can be
	     constructed in the following way.	    
	  Example
               K = abstractSimplicialComplex({{1,2,3,4}, {2,3,5},{1,5}})
          Text
	       The simplex on the vertex set [4] can be constructed as     
          Example
	      L = abstractSimplicialComplex(4)
	  Text
	     The faces and facets of such simplicial complexes can be accessed as
	  Example
	      K_(-1)
	      K_0
	      K_1
	      K_2
	      abstractSimplicialComplexFacets K
	      L_(-1)
	      L_0
	      L_1
	      L_2
	      L_3
	      abstractSimplicialComplexFacets L
///



doc ///
     Key
     	  "How to make reduced and non-reduced simplicial chain complexes"
     Headline
     	  Simplicial homological constructors 
     Description
     	  Text	  
	     Non-reduced and reduced simplicial chain complexes can be constructed in the following way.
	  Example
	       K = abstractSimplicialComplex({{1,2,3,4}, {2,3,5},{1,5}}) 
               k = simplicialChainComplex K
               k.dd
               kRed = reducedSimplicialChainComplex K
               kRed.dd
///

doc ///
     Key
     	  "How to make subsimplical complexes and induced simplicial chain complex maps"
     Headline
     	  Induced simplicial chain complex maps via subsimplicial complexes 
     Description
     	  Text	  
	     Given a subsimplicial complex there are induced simplicial chain complex maps.
	     This is illustrated in the following way.
	  Example
	     K = abstractSimplicialComplex(4,3)
             L = abstractSimplicialComplex(4,2)
             f = inducedSimplicialChainComplexMap(K,L)
             isWellDefined f
             fRed = inducedReducedSimplicialChainComplexMap(K,L)
             isWellDefined fRed
///

doc ///
     Key
     	  "Calculations with random simplicial complexes"
     Headline
     	  Homological calculations on random simplicial complexes
     Description
     	  Text	  
	     In what follows we illustrate a collection of homological calculations that
	     can be performed on random simplicial complexes.
          Text
	     Create a random abstract simplicial complex with vertices supported on a subset of [n] = {1,...,n}.
          Example
	     setRandomSeed(currentTime());
	     K = randomAbstractSimplicialComplex(4)
	     prune HH simplicialChainComplex K
	  Text
	     Create a random simplicial complex on [n] with dimension at most equal to r.
          Example
	     setRandomSeed(currentTime());
	     L = randomAbstractSimplicialComplex(6,3)
	     prune HH simplicialChainComplex L
	  Text
	     Create the random complex Y_d(n,m) which has vertex set
             [n] and complete (d − 1)-skeleton, and has exactly m dimension d faces,
             chosen at random from all binomial(binomial(n,d+1),m) possibilities.
	  Example
	     setRandomSeed(currentTime());
	     M = randomAbstractSimplicialComplex(6,3,2)
	     prune HH simplicialChainComplex M
          Text
	     Creates a random sub-simplicial complex of a given simplicial complex.
          Example
	     setRandomSeed(currentTime());
	     K = randomAbstractSimplicialComplex(4)
	     J = randomSubSimplicialComplex(K)
	     inducedSimplicialChainComplexMap(K,J)
///



--------------------------------------------
-- Documentation of methods and functions --
--------------------------------------------

--
-- Types
--

doc ///
     Key
     	  AbstractSimplicialComplex
     Headline
     	  The type of all abstract simplicial complexes
     Description
     	  Text	  
	     The type AbstractSimplicialComplex is a data type for working with
	     abstract simplicial complexes with vertices supported on [n] = {1,...,n}.
///

doc ///
          Key
       	   (NewMethod, AbstractSimplicialComplex)
///

--
-- Functions and Commands
--

doc ///
    Key
	 (symbol ==,AbstractSimplicialComplex,AbstractSimplicialComplex)
    Headline
         Decide if two simplicial complexes are equal
    Description
          Text
	     Decides if two simplicial complexes are equal.
	  Example
	     randomAbstractSimplicialComplex(4) == randomAbstractSimplicialComplex(4)
///	     

doc ///
    Key
         randomAbstractSimplicialComplex
	 (randomAbstractSimplicialComplex,ZZ)
	 (randomAbstractSimplicialComplex,ZZ,ZZ)
	 (randomAbstractSimplicialComplex,ZZ,ZZ,ZZ)
    Headline
          Create a random simplicial set
    Description
          Text
	     Create a random abstract simplicial complex with vertices supported on a subset of [n] = {1,...,n}.
          Example
	     setRandomSeed(currentTime());
	     K = randomAbstractSimplicialComplex(4)
	  Text
	     Create a random simplicial complex on [n] with dimension at most equal to r.
          Example
	     setRandomSeed(currentTime());
	     L = randomAbstractSimplicialComplex(6,3)
	  Text
	     Create the random complex Y_d(n,m) which has vertex set
             [n] and complete (d − 1)-skeleton, and has exactly m d-dimensional faces,
             chosen at random from all binomial(binomial(n,d+1),m) possibilities.
	  Example
	     setRandomSeed(currentTime());
	     M = randomAbstractSimplicialComplex(6,3,2)
    SeeAlso
        "random"
	"randomSquareFreeMonomialIdeal"
///

doc ///
    Key
         randomSubSimplicialComplex
	 (randomSubSimplicialComplex,AbstractSimplicialComplex)
    Headline
          Create a random sub-simplicial complex
    Description
          Text
	     Creates a random sub-simplicial complex of a given simplicial complex.
          Example
	     setRandomSeed(currentTime());
	     K = randomAbstractSimplicialComplex(4)
	     J = randomSubSimplicialComplex(K)
///


doc ///
     Key
     	  ambientAbstractSimplicialComplex
	  (ambientAbstractSimplicialComplex,AbstractSimplicialComplex)
     Headline
     	  The ambient simplex
     Description
     	  Text	  
	     If an abstract simplicial complex has vertices supported on a subset of [n] = {1,...,n}, and including n,
	     then it seems useful to regard this simplicial complex as being a subsimplicial
	     complex of the simplex on [n].  This method returns this simplex as
	     the ambient simplical complex.
	  Example
	       K = abstractSimplicialComplex({{1,2},{3}})
	       J = ambientAbstractSimplicialComplex(K)
///


doc ///
     Key
     	  ambientAbstractSimplicialComplexSize
	  (ambientAbstractSimplicialComplexSize,AbstractSimplicialComplex)
     Headline
     	  The ambient simplex size
     Description
     	  Text	  
	     If an abstract simplicial complex has vertices supported on a subset of [n] = {1,...,n], and including n,
	     then it seems useful to regard this simplicial complex as being a subsimplicial
	     complex of the simplex on [n].  This method simply returns this largest integer n.
	  Example
	       K = abstractSimplicialComplex({{1,2},{3}})
	       J = ambientAbstractSimplicialComplexSize(K)
///



doc ///
     Key
     	  inducedSimplicialChainComplexMap
	  (inducedSimplicialChainComplexMap,AbstractSimplicialComplex,AbstractSimplicialComplex)
     Headline
     	  Induced maps that arise via inclusions of abstract simplicial complexes
     Description
     	  Text	  
	     If an abstract simplicial complex can be regarded as a subsimplicial complex of another
	     abstract simplicial complex, then it is useful to calculate the induced map at the level of
	     Simplicial Chain Complexes.  This is made
	     possible by the method inducedSimplicialChainComplexMap.
	  Example
	       K = abstractSimplicialComplex({{1,2},{3}})
	       J = ambientAbstractSimplicialComplex(K)
	       inducedSimplicialChainComplexMap(J,K)
	       L = abstractSimplicialComplex {{}}
               inducedSimplicialChainComplexMap(L,L)
	       M = abstractSimplicialComplex {{1}}
	       L = abstractSimplicialComplex {{}}
	       inducedSimplicialChainComplexMap(M,L)
     SeeAlso
          "inducedReducedSimplicialChainComplexMap"             
///

doc ///
     Key
     	  inducedReducedSimplicialChainComplexMap
	  (inducedReducedSimplicialChainComplexMap,AbstractSimplicialComplex,AbstractSimplicialComplex)
     Headline
     	  Induced maps that arise via inclusions of abstract simplicial complexes
     Description
     	  Text	  
	     If an abstract simplicial complex can be regarded as a subsimplicial complex of another
	     abstract simplicial complex, then it is useful to calculate the induced map at the level of
	     Reduced Simplicial Chain Complexes.  This is made
	     possible by the method inducedReducedSimplicialChainComplexMap.
	  Example
	       K = abstractSimplicialComplex({{1,2},{3}})
	       J = ambientAbstractSimplicialComplex(K)
	       inducedReducedSimplicialChainComplexMap(J,K)
               L = abstractSimplicialComplex {{}}
               inducedReducedSimplicialChainComplexMap(L,L)
	       M = abstractSimplicialComplex {{1}}
	       L = abstractSimplicialComplex {{}}
	       inducedReducedSimplicialChainComplexMap(M,L)
     SeeAlso
          "inducedSimplicialChainComplexMap"             
///



doc ///
     Key
     	  reducedSimplicialChainComplex
	  (reducedSimplicialChainComplex,AbstractSimplicialComplex)
     Headline
     	  The reduced homological chain complex that is determined by an abstract simplicial complex 
     Description
     	  Text	  
	     This method returns the reduced homological chain complex (i.e., there is a nonzero term in
		 homological degree -1 that corresponds to the empty face) that is associated
	     to an abstract simplicial complex.  The chain complex is defined over the integers.
          Example
	       K = abstractSimplicialComplex({{1,2,3},{2,4,9},{1,2,3,5,7,8},{3,4}})
	       reducedSimplicialChainComplex(K)
///

doc ///
     Key
     	  simplicialChainComplex
	  (simplicialChainComplex,AbstractSimplicialComplex)
     Headline
     	  The non-reduced homological chain complex that is determined by an abstract simplicial complex 
     Description
     	  Text	  
	     This method returns the (non-reduced) homological chain complex (i.e., there is no nonzero term in
		 homological degree -1 that corresponds to the empty face) that is associated
	     to an abstract simplicial complex.  The chain complex is defined over the integers.
	  Example
	       K = abstractSimplicialComplex({{1,2,3},{1,4,5},{2,4,5,7}})
	       C = simplicialChainComplex(K)
///

doc ///
     Key
     	  abstractSimplicialComplex
	  (abstractSimplicialComplex,List)
	  (abstractSimplicialComplex,ZZ)
	  (abstractSimplicialComplex,ZZ,ZZ)
     Headline
     	  The abstractSimplicialComplex that is determined by an abstract simplicial complex 
     Description
     	  Text	  
	     This method returns the AbstractSimplicialComplex that represents a
	     given abstract simplicial complex.
	     The input is either a given collection of generating faces or an integer.
	     These facets need not
	     be facets.  Moreover, the elements of the faces need not be written 
	     in lexiographic order.  When the input is an integer, the output is the
	     corresponding simplex.
	  Example
	       abstractSimplicialComplex({{1,2,3,4}})
	       abstractSimplicialComplex({{4,1,2,3}, {3,2,5},{1,5}})
	       abstractSimplicialComplex(4)
	  Text
	     The simplicial complex on [n] with r-skelton can be constructed as follows.
	  Example
	     abstractSimplicialComplex(4,2)
///

doc ///
     Key
     	  (symbol _, AbstractSimplicialComplex, ZZ)
     Headline
     	  The k faces of a simplicial complex  
     Description
     	  Text	  
	     This method returns the collection of k faces of a given AbstractSimplicialComplex.
	  Example
	       K = abstractSimplicialComplex(3)
	       K_3
	       K_2
	       K_1
	       K_0
	       K_(-1)
///

doc ///
     Key
          abstractSimplicialComplexFacets 
          (abstractSimplicialComplexFacets, AbstractSimplicialComplex)
     Headline
     	  The facets of a simplicial complex 
     Description
     	  Text	  
	     This method returns the collection of facets of a given AbstractSimplicialComplex.
	  Example
	       K = abstractSimplicialComplex(3)
	       abstractSimplicialComplexFacets K
///

doc ///
     Key
          (dim, AbstractSimplicialComplex)
     Headline
     	  The dimension of a simplicial complex  
     Description
     	  Text	  
	     This method returns the dimension a given AbstractSimplicialComplex.
	  Example
	       K = abstractSimplicialComplex(3)
	       dim K
///



doc ///
          Key
       	   (describe, AbstractSimplicialComplex)
          Headline
	       real description
     	  Usage
	       describe S
	  Description
	       Text
	       	   see describe
    	  SeeAlso
	      	describe	      
///



-* Test section *-
TEST /// -* [insert short title for this test] *-
assert(
K = abstractSimplicialComplex({{1,2},{3}});
J = ambientAbstractSimplicialComplex(K);
isWellDefined inducedReducedSimplicialChainComplexMap(J,K)
    )
assert(
K = abstractSimplicialComplex({{1,2},{3}});
J = ambientAbstractSimplicialComplex(K);
isWellDefined inducedSimplicialChainComplexMap(J,K)
    )
assert(
L = abstractSimplicialComplex({{}});
isWellDefined inducedReducedSimplicialChainComplexMap(L,L)
    )
assert(
M = abstractSimplicialComplex {{1}};
L = abstractSimplicialComplex {{}};
isWellDefined inducedReducedSimplicialChainComplexMap(M,L)
)

-- may have as many TEST sections as needed
///

end--

-* Development section *-

--
--

restart
uninstallPackage "AbstractSimplicialComplexes"
installPackage("AbstractSimplicialComplexes", RemakeAllDocumentation => true)
check "AbstractSimplicialComplexes"
viewHelp"AbstractSimplicialComplexes"


