# (c) Copyright 2010, 2015. CodeWeavers, Inc.

import os

import distversion

import bottlewrapperbase
import bottlequery
import bottlemanagement
import c4profiles
import cxobjc
import cxutils

# for localization
from cxutils import cxgettext as _

class BottleWrapper(bottlewrapperbase.BottleWrapperBase):

    STATUS_INIT = _(u"Scanning\u2026")
    STATUS_UPGRADE = _(u"Needs upgrade")
    STATUS_READY = _(u"Ready")
    STATUS_ARCHIVING = _(u"Archiving\u2026")
    STATUS_DEFAULTING = _(u"Making default\u2026")
    STATUS_RENAMING = _(u"Renaming\u2026")
    STATUS_DELETING = _(u"Deleting\u2026")
    STATUS_DOWNING = _(u"Shutting down\u2026")
    STATUS_FORCE_DOWNING = _(u"Forcing shutdown\u2026")
    STATUS_DOWN = _(u"Shut down")
    STATUS_UPGRADING = _(u"Upgrading\u2026")
    STATUS_REPAIRING = _(u"Repairing\u2026")

    STATUS_X11_UNKNOWN = "X11Uknown"
    STATUS_X11_YES = "X11Yes"
    STATUS_X11_NO = "X11No"

    STATUS_CSMT_UNKNOWN = "CSMTUnknown"
    STATUS_CSMT_ENABLED = "CSMTEnabled"
    STATUS_CSMT_DISABLED = "CSMTDisabled"

    STATUS_DXVK_UNKNOWN = "DXVKUnknown"
    STATUS_DXVK_ENABLED = "DXVKEnabled"
    STATUS_DXVK_CUSTOM = "DXVKCustom"
    STATUS_DXVK_DISABLED = "DXVKDisabled"

    STATUS_HIGHRES_UNKNOWN = "HIGHRESUnknown"
    STATUS_HIGHRES_ENABLED = "HIGHRESEnabled"
    STATUS_HIGHRES_DISABLED = "HIGHRESDisabled"

    STATUS_ESYNC_UNKNOWN = "ESYNCUnknown"
    STATUS_ESYNC_ENABLED = "ESYNCEnabled"
    STATUS_ESYNC_DISABLED = "ESYNCDisabled"

    # versions of DXVK shipped with CrossOver before tracking hashes
    _builtin_dxvk_hashes = {
        # v1.5.1-4 32-bit
        '48246b47cde50ba04e38430cfdfd6e5c', # d3d10_1.dll
        '6277a1e86a31e5fbd2349dbef86813e6', # d3d10core.dll
        '36e24c18280bd8043587579b7c4d0696', # d3d10.dll
        '5760643e45ff5c71c83c72baf5f571c4', # d3d11.dll
        '009e408b22a9ebc18c9578ec3e892023', # d3d9.dll
        '54872eb9af613ba6291029a70d2a08ae', # dxgi.dll
        '4942e3b9367f965cf9c74f00d1bdb6b1', # dxvk_config.dll
        # v1.5.1-4 64-bit
        '88f0d76caf002468487b567242b27149', # d3d10_1.dll
        '9eb17ee84e7acb7eced92fa689d03758', # d3d10core.dll
        'ff5c05e21857f539aeb1bc9bf96e841c', # d3d10.dll
        '82987e5d442a0986cba04cc6329029a1', # d3d11.dll
        '6b08f7eed09265774080e72e2013ab9b', # d3d9.dll
        '548c80385d1f2401c22ab9a2323c4bf5', # dxgi.dll
        '5af43ef518bac478977121eb0152797e', # dxvk_config.dll
        # v1.5.1-7 32-bit
        'e0072f1545f27836dd48bfa712b1d619', # d3d10_1.dll
        '7fe0d348974b50164297f82feba30c44', # dd10core.dll
        '3de7f126f725ccbb6046c4f1234b618a', # d3d10.dll
        'd4ce37a3bd5b45f09656d39b273f7272', # d3d11.dll
        '45181a5c146d2f1f2c3c61e705cf06d5', # d3d9.dll
        '1652b531ab1e0223d5c617bcfcec08fd', # dxgi.dll
        # v1.5.1-7 64-bit
        '6ea7f975b7adf3e1cef5e29a51b53b6d', # d3d10_1.dll
        '2e7c7b7e876fd211ff84c5aa685de1c2', # d3d10core.dll
        '83bdf2ae6202f0a1bac769caad678e5d', # d3d10.dll
        'ac91a7d58b852968fb78499cb3c564d2', # d3d11.dll
        '66baefc5956f938ffe47cbc492800712', # d3d9.dll
        '0ca6075be93f3f203d03ae861b4c9a18', # dxgi.dll
        # v1.5.1-8 32-bit
        '89b1a666b5b35d3f81d42eeb6bf9ddb8', # d3d10_1.dll
        'a569a5651616b8afe03a7c860b3999e9', # d3d10core.dll
        '2bc0c327134fa0cabf4651f79b05744a', # d3d10.dll
        '6b654ecc01b69fd0d3958b74e6e18278', # d3d11.dll
        '092748859b2b297a7f832661c1675522', # d3d9.dll
        'f261ea9fba28b32fa20241e100aa0695', # dxgi.dll
        '4942e3b9367f965cf9c74f00d1bdb6b1', # dxvk_config.dll
        # v1.5.1-8 64-bit
        '67814b09774e6f1342b85b3fbda0e5ca', # d3d10_1.dll
        '217e556b7c1f03f2c13ca483590fa67a', # d3d10core.dll
        'b4a0ab5045ebf7618ea51969b6394bd7', # d3d10.dll
        '37c6d59416531ed250b417749dba25bd', # d3d11.dll
        '30c064928b28231a722806093a5577bf', # d3d9.dll
        '3255409c9699ff7c78707198fe4bdf40', # dxgi.dll
        '5af43ef518bac478977121eb0152797e', # dxvk_config.dll
    }

    def __init__(self, inName):
        cxutils.expect_unicode(inName)
        bottlewrapperbase.BottleWrapperBase.__init__(self, inName)
        self._change_delegates = []
        self._bottle_info_ready = False
        self._property_dict = {}
        self.current_description = None
        self.bottletemplate = ""
        self.arch = ""
        self._needs_refresh_up_to_date = True
        self._last_quit_failed = False

        self.control_panel_loading = False
        self._control_panel_ready = False
        self._control_panel_table = []
        self._control_panel_off_thread_table = []


        self.is_x11_driver_ready = False
        self.is_x11_driver_state = BottleWrapper.STATUS_X11_UNKNOWN
        self._graphicsDriverKey = "HKEY_CURRENT_USER\\Software\\Wine\\Drivers"

        self.is_csmt_disabled_ready = False
        self.is_csmt_disabled_loading = False
        self.is_csmt_disabled_state = BottleWrapper.STATUS_CSMT_UNKNOWN
        self._direct3DKey = "HKEY_CURRENT_USER\\Software\\Wine\\Direct3D"

        self.is_dxvk_enabled_ready = False
        self.is_dxvk_enabled_loading = False
        self._dlloverridesKey = "HKEY_CURRENT_USER\\Software\\Wine\\DllOverrides"
        self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_UNKNOWN

        self.is_high_resolution_enabled_ready = False
        self.is_high_resolution_enabled_loading = False
        self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_UNKNOWN

        self.is_esync_enabled_ready = False
        self.is_esync_enabled_loading = False
        self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_UNKNOWN

        self.is_high_resolution_ready = False
        self.is_high_resolution_state = False
        self._macDriverKey = "HKEY_CURRENT_USER\\Software\\Wine\\Mac Driver"
        self._logPixelsKey = "HKEY_CURRENT_USER\\Control Panel\\Desktop"
        self._oldLogPixelsKey = "HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Hardware Profiles\\Current\\Software\\Fonts"

        self._installed_packages = {}
        self.installedPackageValues = []
        self._installed_packages_off_thread = {}
        self._installed_packages_ready = False
        self.installed_packages_loading = False

        self.status = BottleWrapper.STATUS_INIT
        self.status_overrides = []

        # mtime values:
        #  0 = not yet updated
        #  -1 = no config file
        #  None = deleted
        #  anything else = mtime of config file
        self.mtime = 0

        # This is something we need to know immediately for
        #  Mac initialization purposes. Hopefully it's not too
        #  expensive to do here.
        self.is_managed = bottlequery.is_managed(self.name)
        self.refresh_up_to_date()


        self.marked_for_death = False

        # Let's learn a bit about ourselves. Everything here
        #  should be very quick.
        self.wine_prefix = bottlequery.get_prefix_for_bottle(self.name)
        self.system_drive = os.path.join(self.wine_prefix, 'drive_c')
        self.is_default = (self.name == bottlequery.get_default_bottle())

        self.pre_load_basic_info()
        self.load_basic_info()
        self.post_load_basic_info()


    # This will be called, below, only on ObjC. It sets up
    #  some miscellaneous notifications that the GUI needs.
    @classmethod
    def prepare_kvo_notification(cls):
        # The setKeys_... functions are implemented in pyobjc and thus
        # invisible to pylint on non-Mac platforms.
        # pylint: disable=E1101
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["up_to_date", "is_managed", "is_busy"], "can_run_commands")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["name", "is_default"], "display_name")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["name"], "changeablename")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["status", "up_to_date", "bottle_info_ready"], "is_active")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["up_to_date", "is_managed", "marked_for_death"], "can_edit")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["is_managed", "marked_for_death", "status"], "canInstall")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["status", "_last_quit_failed"], "can_force_quit")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["status", "marked_for_death"], "is_busy")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["status", "marked_for_death", "is_managed"], "can_rename")
        cls.setKeys_triggerChangeNotificationsForDependentKey_(["up_to_date", "is_managed"], "needs_upgrade")



    #####
    #
    # load basic info
    #
    #####

    @cxobjc.namedSelector(b'preLoadBasicInfo')
    def pre_load_basic_info(self):
        """This function is pretty fast and MUST be called in the main thread,
        BEFORE calling load_basic_info().
        """
        self.add_status_override(BottleWrapper.STATUS_INIT)
        self.bottle_info_ready = False


    @cxobjc.namedSelector(b'loadBasicInfo')
    def load_basic_info(self):
        """This function loads the bottle's basic properties.

        It takes quite a bit of time to run and thus must be run in the
        background. Thus, for purposes of thread management, it is divided into
        three parts, with pre_ and post_ functions that are cheap and need to
        be called on the main thread, in order to properly arrange the state of
        our properties.
        """
        self.set_property_dict(bottlequery.get_bottle_properties(self.name))


    @cxobjc.namedSelector(b'postLoadBasicInfo')
    def post_load_basic_info(self):
        """This function is pretty fast and MUST be called in the main thread,
        AFTER load_basic_info() has returned.
        """
        if self._property_dict:
            # copy everything we've learned out of the
            #  property dict and into actual properties.
            self.bottletemplate = self._property_dict["template"]

            if "managed" in self._property_dict:
                self.is_managed = self._property_dict["managed"]
            else:
                self.is_managed = False

            self.arch = self._property_dict["arch"]

            self.current_description = self._property_dict["description"]

            # Notify everyone that we're ready to go.
            self.bottle_info_ready = True
        else:
            self.bottle_info_ready = False

        self.remove_status_override(BottleWrapper.STATUS_INIT)

    def _maybe_upgrade(self):
        # If we get the control panel or installed application info for a
        # bottle that needs to be upgraded, this upgrades the bottle, which
        # in turn updates the mtime of cxbottle.conf. This would cause us to
        # incorrectly assume our information is out of date, so if we need
        # an upgrade we trigger it explicitly and ignore the mtime change.
        stub_needs_upgrade = False
        if self.is_managed:
            stub_needs_upgrade = not bottlemanagement.get_up_to_date(self.name, "private")

        if not self.up_to_date or stub_needs_upgrade:
            args = [os.path.join(cxutils.CX_ROOT, "bin", "wine"),
                    "--bottle", self.name, '--no-gui',
                    "--ux-app", "true"]
            cxutils.system(args)
            self.mtime = self._real_mtime()
            self._needs_refresh_up_to_date = True

    #####
    #
    # load x11 driver preferences
    #
    # Three sections, same as load_basic_info, above.
    #
    # Loads the setting which determines whether
    # the x11 driver is enabled or not.
    #
    #####

    @cxobjc.namedSelector(b'preLoadIsX11Driver')
    def pre_load_is_x11_driver(self):
        self.is_x11_driver_ready = False

    @cxobjc.namedSelector(b'loadIsX11Driver')
    def load_is_x11_driver(self):

        self.is_x11_driver_state = BottleWrapper.STATUS_X11_NO



        try:
            _subkeys, values = bottlequery.get_registry_key(self.name,
                                                            self._graphicsDriverKey)
            if 'graphics' in values:
                if values['graphics'].upper().find("X11") == 0:
                    self.is_x11_driver_state = BottleWrapper.STATUS_X11_YES
        except bottlequery.NotFoundError:
            pass

    @cxobjc.namedSelector(b'postLoadIsX11Driver')
    def post_load_is_x11_driver(self):
        self.is_x11_driver_ready = True

    @cxobjc.namedSelector(b'enableX11Driver')
    def enable_x11_driver(self):
        bottlequery.set_registry_key(self.name,
                                     self._graphicsDriverKey,
                                     "Graphics",
                                     "x11,mac")
        self.is_x11_driver_state = BottleWrapper.STATUS_X11_YES

    @cxobjc.namedSelector(b'disableX11Driver')
    def disable_x11_driver(self):
        bottlequery.unset_registry_value(self.name,
                                         self._graphicsDriverKey,
                                         "Graphics")
        self.is_x11_driver_state = BottleWrapper.STATUS_X11_NO

    def is_x11_driver(self):
        return self.is_x11_driver_state == BottleWrapper.STATUS_X11_YES

    #####
    #
    # load dxvk preferences
    #
    # Loads the setting which determines whether
    # dxvk is enabled in this bottle.
    #
    #####

    def get_dxvkfiles_filename(self):
        return bottlequery.get_dxvkfiles_filename(self.name)

    def get_dxvkfiles_config(self):
        return bottlequery.get_dxvkfiles_config(self.name)

    @cxobjc.namedSelector(b'preLoadIsDXVKEnabled')
    def pre_load_is_dxvk_enabled(self):
        self.is_dxvk_enabled_ready = False
        self.is_dxvk_enabled_loading = True

    @cxobjc.namedSelector(b'loadIsDXVKEnabled')
    def load_is_dxvk_enabled(self):

        self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_UNKNOWN

        try:

            self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_DISABLED

            _subkeys, values = bottlequery.get_registry_key(self.name,
                                                            self._dlloverridesKey)

            if 'd3d11' in values:
                if values['d3d11'].startswith('native'):
                    self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_CUSTOM
                    d3d11_winpath = bottlequery.expand_win_string(bottlequery.get_win_environ(self.name, c4profiles.ENVIRONMENT_VARIABLES), "%WinSysDir64%\\d3d11.dll")
                    d3d11_path = bottlequery.get_native_path(self.name, d3d11_winpath)
                    checksum = cxutils.md5sum_path(d3d11_path)

                    if checksum in self._builtin_dxvk_hashes or \
                        checksum == self.get_dxvkfiles_config()['DxvkFiles'].get(d3d11_winpath, ''):
                        self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_ENABLED
        except bottlequery.NotFoundError:
            pass

    @cxobjc.namedSelector(b'postLoadIsDXVKEnabled')
    def post_load_is_dxvk_enabled(self):
        self.is_dxvk_enabled_ready = True
        self.is_dxvk_enabled_loading = False

    @cxobjc.namedSelector(b'enableDXVK')
    def enable_dxvk(self):
        bottlequery.enable_dxvk(self.name)
        self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_ENABLED

    @cxobjc.namedSelector(b'disableDXVK')
    def disable_dxvk(self):
        bottlequery.disable_dxvk(self.name)
        self.is_dxvk_enabled_state = BottleWrapper.STATUS_DXVK_DISABLED

    def is_dxvk_enabled(self):
        return self.is_dxvk_enabled_state == BottleWrapper.STATUS_DXVK_ENABLED


    @cxobjc.namedSelector(b'preLoadIsCSMTDisabled')
    def pre_load_is_csmt_disabled(self):
        self.is_csmt_disabled_ready = False
        self.is_csmt_disabled_loading = True

    @cxobjc.namedSelector(b'loadIsCSMTDisabled')
    def load_is_csmt_disabled(self):

        self.is_csmt_disabled_state = BottleWrapper.STATUS_CSMT_ENABLED

        try:
            _subkeys, values = bottlequery.get_registry_key(self.name,
                                                            self._direct3DKey)

            if 'csmt' in values:
                if isinstance(values['csmt'], int):
                    if values['csmt']:
                        self.is_csmt_disabled_state = BottleWrapper.STATUS_CSMT_ENABLED
                    else:
                        self.is_csmt_disabled_state = BottleWrapper.STATUS_CSMT_DISABLED
                elif values['csmt'] == 'disabled':
                    self.is_csmt_disabled_state = BottleWrapper.STATUS_CSMT_DISABLED
        except bottlequery.NotFoundError:
            pass

    @cxobjc.namedSelector(b'postLoadIsCSMTDisabled')
    def post_load_is_csmt_disabled(self):
        self.is_csmt_disabled_ready = True
        self.is_csmt_disabled_loading = False

    @cxobjc.namedSelector(b'enableCSMT')
    def enable_csmt(self):
        bottlequery.set_registry_key(self.name,
                                     self._direct3DKey,
                                     "CSMT",
                                     'dword:1')
        self.is_csmt_disabled_state = BottleWrapper.STATUS_CSMT_ENABLED

    @cxobjc.namedSelector(b'disableCSMT')
    def disable_csmt(self):
        bottlequery.set_registry_key(self.name,
                                     self._direct3DKey,
                                     "CSMT",
                                     'dword:0')
        self.is_csmt_disabled_state = BottleWrapper.STATUS_CSMT_DISABLED

    def is_csmt_disabled(self):
        return self.is_csmt_disabled_state == BottleWrapper.STATUS_CSMT_DISABLED


    @cxobjc.python_method
    def pre_load_is_high_resolution_enabled(self):
        self.is_high_resolution_enabled_ready = False
        self.is_high_resolution_enabled_loading = True

    @cxobjc.python_method
    def load_is_high_resolution_enabled(self):

        self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_DISABLED

        values = {}
        try:
            _subkeys, values = bottlequery.get_registry_key(self.name,
                                                            self._logPixelsKey)
        except bottlequery.NotFoundError:
            pass
        if 'logpixels' not in values:
            try:
                _subkeys, values = bottlequery.get_registry_key(self.name,
                                                                self._oldLogPixelsKey)
            except bottlequery.NotFoundError:
                pass

        if values.get('logpixels', 96) >= 110:
            self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_ENABLED

    @cxobjc.python_method
    def post_load_is_high_resolution_enabled(self):
        self.is_high_resolution_enabled_ready = True
        self.is_high_resolution_enabled_loading = False

    @cxobjc.python_method
    def enable_high_resolution(self, logpixels):
        bottlequery.set_registry_key(self.name,
                                     self._logPixelsKey,
                                     "LogPixels",
                                     "dword:" + hex(logpixels)[2:])
        self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_ENABLED

    @cxobjc.python_method
    def disable_high_resolution(self):
        # Force the DPI to 96 here so winewrapper won't overwrite it.
        bottlequery.set_registry_key(self.name,
                                     self._logPixelsKey,
                                     "LogPixels",
                                     "dword:60")
        self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_DISABLED

    @cxobjc.python_method
    def is_high_resolution_enabled(self):
        return self.is_high_resolution_enabled_state == BottleWrapper.STATUS_HIGHRES_ENABLED


    @cxobjc.namedSelector(b'preLoadIsESyncEnabled')
    def pre_load_is_esync_enabled(self):
        self.is_esync_enabled_ready = False
        self.is_esync_enabled_loading = True

    @cxobjc.namedSelector(b'loadIsESyncEnabled')
    def load_is_esync_enabled(self):
        self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_DISABLED

        enabled = bottlequery.get_config_value(
            self.name, "EnvironmentVariables", "WINEESYNC", "0") != "0"
        if enabled:
            self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_ENABLED

    @cxobjc.namedSelector(b'postLoadIsESyncEnabled')
    def post_load_is_esync_enabled(self):
        self.is_esync_enabled_ready = True
        self.is_esync_enabled_loading = False

    @cxobjc.namedSelector(b'enableESync')
    def enable_esync(self):
        bottlequery.set_config_value(
            self.name, "EnvironmentVariables", "WINEESYNC", "1")
        self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_ENABLED

    @cxobjc.namedSelector(b'disableESync')
    def disable_esync(self):
        bottlequery.set_config_value(
            self.name, "EnvironmentVariables", "WINEESYNC", "0")
        self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_DISABLED

    def is_esync_enabled(self):
        return self.is_esync_enabled_state == BottleWrapper.STATUS_ESYNC_ENABLED


    #####
    #
    # load Mac driver high resolution mode preferences
    #
    #####

    def preLoadIsHighResolution(self):
        self.is_high_resolution_ready = False

    def loadIsHighResolution(self):

        self.is_high_resolution_state = False

        try:
            _subkeys, values = bottlequery.get_registry_key(self.name,
                                                            self._macDriverKey)
            if 'retinamode' in values:
                if values['retinamode'][0] in set("yYtT1"):
                    self.is_high_resolution_state = True
        except bottlequery.NotFoundError:
            pass

    def postLoadIsHighResolution(self):
        self.is_high_resolution_ready = True

    def enableHighResolution(self):
        if bottlequery.set_registry_key(self.name, self._macDriverKey, "RetinaMode", "y"):
            self.is_high_resolution_state = True
            values = {}
            try:
                _subkeys, values = bottlequery.get_registry_key(self.name,
                                                                self._logPixelsKey)
            except bottlequery.NotFoundError:
                pass
            if 'logpixels' not in values:
                try:
                    _subkeys, values = bottlequery.get_registry_key(self.name,
                                                                    self._oldLogPixelsKey)
                except bottlequery.NotFoundError:
                    pass
            logpixels = values.get('logpixels', 96)
            logpixels *= 2
            bottlequery.set_registry_key(self.name,
                                         self._logPixelsKey,
                                         "LogPixels",
                                         "dword:" + hex(logpixels)[2:])

    def disableHighResolution(self):
        if bottlequery.unset_registry_value(self.name, self._macDriverKey, "RetinaMode"):
            self.is_high_resolution_state = False
            values = {}
            try:
                _subkeys, values = bottlequery.get_registry_key(self.name,
                                                                self._logPixelsKey)
            except bottlequery.NotFoundError:
                pass
            if 'logpixels' not in values:
                try:
                    _subkeys, values = bottlequery.get_registry_key(self.name,
                                                                    self._oldLogPixelsKey)
                except bottlequery.NotFoundError:
                    pass
            if 'logpixels' in values:
                logpixels = values['logpixels']
                logpixels = max(logpixels // 2, 96)
                bottlequery.set_registry_key(self.name,
                                             self._logPixelsKey,
                                             "LogPixels",
                                             "dword:" + hex(logpixels)[2:])

    def is_high_resolution(self):
        return self.is_high_resolution_state


    #####
    #
    # load control panel info
    #
    # Three sections, same as load_basic_info, above.
    #
    # Loads the list of control panels available for this bottle,
    # and sets up icons and other user-readable-data for each.
    #
    #####

    @cxobjc.namedSelector(b'preLoadControlPanelInfo', b'v@:')
    def pre_load_control_panel_info(self):
        """This function is pretty fast and MUST be called in the main thread,
        BEFORE calling load_control_panel_info().
        """
        self.control_panel_loading = True
        self.control_panel_ready = False
        self.add_status_override(BottleWrapper.STATUS_INIT)

    @cxobjc.namedSelector(b'loadControlPanelInfo', b'v@:')
    def load_control_panel_info(self):
        """This function initializes the bottle's list of control panel applets.

        It takes quite a bit of time to run and thus must be run in the
        background. Thus, for purposes of thread management, it is divided into
        three parts, with pre_ and post_ functions that are cheap and need to
        be called on the main thread, in order to properly arrange the state of
        our properties.
        """
        self._maybe_upgrade()

        self._control_panel_off_thread_table = bottlequery.get_control_panel_info(self.name)

        if not distversion.IS_MACOSX:
            self._control_panel_off_thread_table.append(["cxassoceditui", _("Edit Associations"),
                                                         _("Manage the Windows programs used to open files in the native environment"), "cxassocedit"])

            self._control_panel_off_thread_table.append(["cxmenueditui", _("Edit Menus"),
                                                         _("Manage the menus and desktop icons in this bottle"), "cxmenuedit"])


    @cxobjc.namedSelector(b'postLoadControlPanelInfo', b'v@:')
    def post_load_control_panel_info(self):
        """This function is pretty fast and MUST be called in the main thread,
        AFTER load_control_panel_info() has returned.
        """
        if self._needs_refresh_up_to_date:
            self.refresh_up_to_date()

        icon_sizes = ['48x48', '24x24', '64x64', '32x32', '']
        if distversion.IS_MACOSX:
            icon_sizes.insert(0, '96x96')

        cptable = []
        std_root = os.path.join(cxutils.CX_ROOT, 'share', 'icons')
        for panel in self._control_panel_off_thread_table:
            cpdict = {}
            cpdict["exe"] = panel[0]
            cpdict["name"] = panel[1]
            cpdict["description"] = panel[2]
            base, ext = os.path.splitext(panel[3])
            if ext == '' and base != '':
                # This is a builtin icon
                cpdict["icon"] = cxutils.get_icon_path(std_root, '', base, icon_sizes)
            elif distversion.IS_MACOSX:
                # conveniently, cxcplinfo provides us with .ico files in
                # addition to the .xpms which the Mac can't handle.
                if ext.lower() == ".xpm":
                    ext = ".ico"
                cpdict["icon"] = base + ext
            else:
                cpdict["icon"] = panel[3]
            if cpdict["icon"] is None or cpdict["icon"] == '':
                cpdict["icon"] = cxutils.get_icon_path(std_root, '', 'crossover', icon_sizes)
            cptable.append(cpdict)
        self.control_panel_table = cptable
        self.control_panel_ready = True
        self.control_panel_loading = False
        self.remove_status_override(BottleWrapper.STATUS_INIT)


    @cxobjc.namedSelector(b'launchControlPanelApplet:')
    def launch_control_panel_applet(self, applet):
        wine = os.path.join(cxutils.CX_ROOT, "bin", "wine")
        if applet.lower().endswith(".exe.so"):
            args = [wine, "--bottle", self.name, "--wl-app", applet]
            if applet.lower() == "reboot.exe.so":
                args.append("--show-gui")
        elif applet.lower().endswith(".cpl"):
            args = [wine, "--bottle", self.name, "--wl-app", "rundll32", "shell32.dll,Control_RunDLL", applet]
        else:
            __import__(applet).start(self)
            return

        cxutils.run(args, background=True)


    #####
    # load installed applications
    #
    # Three sections, same as load_basic_info, above.
    #
    #
    # Loads the list of applications installed in the bottle.
    # This can take quite a while.
    #
    #####
    @cxobjc.namedSelector(b'preLoadInstalledApplications', b'v@:')
    def pre_load_installed_applications(self):
        """This function is pretty fast and MUST be called in the main thread,
        BEFORE calling load_installed_applications().
        """
        self.installed_packages_ready = False
        self.installed_packages_loading = True
        self._installed_packages_off_thread = {}
        self.add_status_override(BottleWrapper.STATUS_INIT)

    @cxobjc.namedSelector(b'loadInstalledApplications', b'v@:')
    def load_installed_applications(self):
        """This function determines the list of applications that are installed
        in the bottle.

        It takes quite a bit of time to run and thus must be run in the
        background. Thus, for purposes of thread management, it is divided into
        three parts, with pre_ and post_ functions that are cheap and need to
        be called on the main thread, in order to properly arrange the state of
        our properties.
        """
        import c4profilesmanager
        import appdetector

        self._maybe_upgrade()

        profiles = c4profilesmanager.C4ProfilesSet.all_profiles()
        self._installed_packages_off_thread = appdetector.fast_get_installed_applications(self.name, profiles)

    @cxobjc.namedSelector(b'postLoadInstalledApplications', b'v@:')
    def post_load_installed_applications(self):
        """This function is pretty fast and MUST be called in the main thread,
        AFTER load_installed_applications() has returned.
        """
        if self._needs_refresh_up_to_date:
            self.refresh_up_to_date()

        self.installed_packages = self._installed_packages_off_thread
        self.installed_packages_ready = True
        self.installed_packages_loading = False
        self.remove_status_override(BottleWrapper.STATUS_INIT)

    @cxobjc.namedSelector(b'fileIsInBottle:')
    def file_is_in_bottle(self, filename):
        drivepath = os.path.abspath(os.path.realpath(self.system_drive))
        filepath = os.path.abspath(os.path.realpath(filename))
        return filepath.startswith(drivepath)

    @cxobjc.namedSelector(b'writeDescription')
    def write_description(self):
        #  Note:  Expensive!  Should only be done within an operation.
        current_description = self.current_description
        if self._property_dict["description"] != current_description:
            bottlemanagement.set_bottle_description(self.name, current_description)
            self._property_dict["description"] = current_description

    def is_description_written(self):
        return self.current_description == self._property_dict["description"]

    @cxobjc.python_method
    def change_description(self, inDescription):
        self.current_description = inDescription

    @cxobjc.namedSelector(b'preRename')
    def pre_rename(self):
        if self.changeablename == self.name:
            return

        self.add_status_override(BottleWrapper.STATUS_RENAMING)
        self.marked_for_death = True

    def rename(self):
        if self.changeablename == self.name:
            return False, None

        (success, err) = bottlemanagement.rename_bottle(self.name, self.changeablename)
        if not success:
            print("Rename from %s to %s failed:  %s" % (self.name, self.changeablename, err))
        return success, err

    @cxobjc.namedSelector(b'postRename:')
    def post_rename(self, success):
        if self.changeablename == self.name:
            return

        self.remove_status_override(BottleWrapper.STATUS_RENAMING)
        self.marked_for_death = False

        if not success:
            self.changeablename = self.name
            return

        self.name = self.changeablename
        self.wine_prefix = bottlequery.get_prefix_for_bottle(self.name)
        self.system_drive = os.path.join(self.wine_prefix, 'drive_c')

    @cxobjc.python_method
    def get_control_panel_table(self):
        return self._control_panel_table

    @cxobjc.python_method
    def set_control_panel_table(self, in_table):
        self._control_panel_table = in_table
        for delegate in self._change_delegates:
            delegate.bottleChanged(self)

    control_panel_table = property(get_control_panel_table, set_control_panel_table)

    @cxobjc.python_method
    def get_installed_packages(self):
        return self._installed_packages

    @cxobjc.python_method
    def set_installed_packages(self, installed_apps):
        self._installed_packages = installed_apps
        # installedPackageValues stores the same data but as an array.
        # We need this for bindings, and having an actual member
        # allows us to use KVO properly.
        self.installedPackageValues = cxutils.values(installed_apps)
        for delegate in self._change_delegates:
            delegate.bottleChanged(self)

    installed_packages = property(get_installed_packages, set_installed_packages)

    @cxobjc.python_method
    def set_property_dict(self, inDict):
        self._property_dict = inDict

        # if current_description varies from the
        #  dict value, we know it's dirty and needs
        #  to be written.
        if "description" in inDict:
            self.current_description = inDict["description"]
        else:
            self.current_description = None

        for delegate in self._change_delegates:
            delegate.bottleChanged(self)

    def _real_mtime(self):
        config_file_path = bottlequery.config_file_path(self.name)
        try:
            mtime = os.path.getmtime(config_file_path)
        except OSError:
            # distinguish between a bottle with no config file and a deleted bottle
            if os.path.exists(self.wine_prefix):
                mtime = -1
            else:
                mtime = None
        return mtime

    def get_needs_update(self, updating=False):
        if self.status_overrides:
            return False

        mtime = self._real_mtime()

        if self.mtime != mtime:
            if updating:
                self.mtime = mtime
            return True
        return False

    needs_update = property(get_needs_update)

    def get_needs_import(self):
        return False

    needs_import = property(get_needs_import)

    @cxobjc.namedSelector(b'addStatusOverride:')
    def add_status_override(self, status):
        self.status_overrides.append(status)
        self.status = status
        for delegate in self._change_delegates:
            delegate.bottleChanged(self)

    @cxobjc.python_method
    def remove_all_status_overrides(self, status):
        while status in self.status_overrides:
            self.remove_status_override(status)

    @cxobjc.namedSelector(b'removeStatusOverride:')
    def remove_status_override(self, status):
        if status in self.status_overrides:
            self.status_overrides.remove(status)
        if self.status_overrides:
            self.status = self.status_overrides[len(self.status_overrides)-1]
        elif not self.bottle_info_ready or not self.installed_packages_ready or \
                not (distversion.IS_MACOSX or self.control_panel_ready):
            self.status = BottleWrapper.STATUS_INIT
        else:
            self.status = BottleWrapper.STATUS_READY
        for delegate in self._change_delegates:
            delegate.bottleChanged(self)

    def get_can_run_commands(self):
        return (self.up_to_date or not self.is_managed) and not self.is_busy

    can_run_commands = property(get_can_run_commands)

    def get_can_edit(self):
        return self.up_to_date and not self.marked_for_death and not self.is_managed

    can_edit = property(get_can_edit)

    def canInstall(self):
        return not self.is_managed and not self.marked_for_death and \
            BottleWrapper.STATUS_RENAMING not in self.status_overrides and \
            BottleWrapper.STATUS_DELETING not in self.status_overrides and \
            BottleWrapper.STATUS_DOWNING not in self.status_overrides and \
            BottleWrapper.STATUS_FORCE_DOWNING not in self.status_overrides and \
            BottleWrapper.STATUS_DOWN not in self.status_overrides

    def get_is_active(self):
        return self.bottle_info_ready and not self.status_overrides and self.up_to_date

    is_active = property(get_is_active)

    def get_can_force_quit(self):
        """Be careful of race conditions with this property.  If all you've done
        is initiate an asynchronous operation to quit the bottle, these statuses
        won't necessarily be set yet."""
        return (self._last_quit_failed or BottleWrapper.STATUS_DOWNING in self.status_overrides) and \
            BottleWrapper.STATUS_FORCE_DOWNING not in self.status_overrides

    can_force_quit = property(get_can_force_quit)

    def get_is_busy(self):
        return self.marked_for_death or \
            BottleWrapper.STATUS_RENAMING in self.status_overrides or \
            BottleWrapper.STATUS_DELETING in self.status_overrides or \
            BottleWrapper.STATUS_REPAIRING in self.status_overrides or \
            BottleWrapper.STATUS_ARCHIVING in self.status_overrides

    is_busy = property(get_is_busy)

    def get_can_rename(self):
        return not self.marked_for_death and not self.status_overrides and not self.is_managed

    can_rename = property(get_can_rename)

    def get_needs_upgrade(self):
        return self.is_managed and not self.up_to_date

    needs_upgrade = property(get_needs_upgrade)

    # For ObjC:
    def setMarkedForDeath_(self, inBool):
        if inBool:
            self.add_status_override(BottleWrapper.STATUS_DELETING)
        self.marked_for_death = inBool

    def is_running(self):
        return bottlemanagement.is_running(self.name)

    def preQuit(self):
        """On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        self._last_quit_failed = False
        self.add_status_override(BottleWrapper.STATUS_DOWN)
        self.add_status_override(BottleWrapper.STATUS_DOWNING)

    def doQuit(self):
        return bottlemanagement.quit_bottle(self.name)

    def postQuit_(self, success):
        """On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        self.remove_status_override(BottleWrapper.STATUS_DOWNING)
        if not success:
            self._last_quit_failed = True
            self.remove_status_override(BottleWrapper.STATUS_DOWN)

    def quit(self):
        self.preQuit()
        success = self.doQuit()
        self.postQuit_(success)
        return success

    @cxobjc.namedSelector(b'refreshUpToDate')
    def refresh_up_to_date(self):
        """On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        self._needs_refresh_up_to_date = False

        if self.is_managed:
            self.up_to_date = bottlemanagement.get_up_to_date(self.name, "managed")
        else:
            self.up_to_date = bottlemanagement.get_up_to_date(self.name)

        if self.is_managed:
            if not self.up_to_date:
                self.add_status_override(BottleWrapper.STATUS_UPGRADE)
            else:
                self.remove_all_status_overrides(BottleWrapper.STATUS_UPGRADE)

    def preForceQuit(self):
        """On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        self._last_quit_failed = False
        self.add_status_override(BottleWrapper.STATUS_DOWN)
        self.add_status_override(BottleWrapper.STATUS_FORCE_DOWNING)

    def doForceQuit(self):
        return bottlemanagement.kill_bottle(self.name)

    def postForceQuit_(self, success):
        """On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        self.remove_status_override(BottleWrapper.STATUS_FORCE_DOWNING)
        if not success:
            self.remove_status_override(BottleWrapper.STATUS_DOWN)

    def force_quit(self):
        self.preForceQuit()
        success = self.doForceQuit()
        self.postForceQuit_(success)
        return success

    # Cancel shutdown is called to restore the status of a bottle
    #  to a non-deleting, non-dying, non-shutting-down status.
    def cancelShutdown(self):
        """On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        self.remove_status_override(BottleWrapper.STATUS_DOWN)
        self.remove_status_override(BottleWrapper.STATUS_FORCE_DOWNING)
        self.remove_status_override(BottleWrapper.STATUS_DOWNING)
        self.remove_status_override(BottleWrapper.STATUS_DELETING)

    @cxobjc.namedSelector(b'displayName')
    def get_display_name(self):
        if self.is_default:
            return _("%s (default)") % self.name
        return self.name

    display_name = property(get_display_name)

    def get_appid(self):
        return bottlequery.get_appid(self.name)

    appid = property(get_appid)

    # A couple of crutches for reverse-compatibility with the old BottleWrapper.m
    def bottleName(self):
        return self.name


if distversion.IS_MACOSX:
    BottleWrapper.prepare_kvo_notification()
