/*****************************************************************************
 * yahoophshare.c
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * Copyright (C) 2006 Stefan Sikora
 * Copyright (C) 2008 Gregory D Hosler
 *****************************************************************************/
 
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>

#include <gtk/gtk.h>

#include "config.h"

#include "gyachi_md5.h"
#include "gyach.h"
#include "main.h"
#include "yahoochat.h"
#include "util.h"
#include "yahoophshare.h"
#include "yahoofxfer.h"
#include "interface.h"
#include "sounds.h"
#include "friends.h"
#include "profname.h"
#include "conference.h"
#include "users.h"
#include "interface.h"
#include "pmwindow.h"
#include "packet_handler.h"

#include "gyachi_lib.h"
#include "gy_config.h"
#include "theme_support.h"

#include <jpeglib.h>

/* photoshare session state information. 1 of these per session/user */
typedef struct _PHOTOSHARE {
	GtkWidget *vbox;
	GtkWidget *hscrollbox;
	GtkWidget *preview_image_frame;
	GtkWidget *counterlabel;
	GtkWidget *scrollbox;
	GtkWidget *preview_image;
	GtkWidget *preview_image_canvas;
	GtkWidget *preview_image_ebox;
	GtkWidget *friend_view_image;
	GtkWidget *popup_menu;
	gint       picturecount;
	gint       picture_sent_count;
	GList     *picturelist;
	GList     *current_thumb;
	char      *buddy_filekey;
	int        friendview_button_state;
	GtkWidget *button_friendsview;
} PHOTOSHARE;

/* photoshare photo state information. 1 of these per photo.
 * These are linked into the picturelist in the user's PHOTOSHARE struct
 * and are linked into the photo session's hscrollbox.
 */
typedef struct {
	char *filekey;
	char *filename;
	char *extn;
	char *hint;
	char *file_token;
	char *preview;
	int filesize;
	int height;
	int width;
	int rotation;
	int last_size;
	GList *my_element;
	GtkWidget *button;
	GtkWidget *bar;
	GtkWidget *popup_menu;
	PHOTOSHARE_DIR direction;
	gdouble h_adj;
	gdouble v_adj;

	/* Below for pointer support */
	int in_image;
	int pointer_supported;
	int pointer_on;
	int pointer_x;
	int pointer_y;
} PHOTO_INFO;

typedef struct {
	char *host_ip;
	char *who;
	PHOTO_INFO *photo_info;
} FILE_SEND_INFO;

typedef struct {
	PHOTO_INFO *photo_info;
	PM_SESSION *pm_sess;
} FILE_RECEIVE_INFO;

typedef struct {
	PHOTO_INFO *photo_info;
	PM_SESSION *pm_sess;
} SENDFILE_QUEUE_INFO;

GList *sendfile_queue = NULL;

char *xpm_black[]= {
  "48 5 1 1",
  "x	c black",
  "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx",
  "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx",
  "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx",
  "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx",
  "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"};

GdkCursor *yphoto_cursor_tcross    = NULL;

/* forward declaration */
void yphoto_add_file(PM_SESSION *pm_sess, char *path, char *hint, char *filekey);


static const char *yphoto_b64s = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=-";

/* yahoo_b64 "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/="
 * mac 64    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789._-"
 */
void b64_to_mac64(char *target)
{
	char *ptr;

	ptr=target;
	while ((ptr=strchr(ptr, '+'))) {
		*ptr = '.';
	}
	ptr=target;
	while ((ptr=strchr(ptr, '/'))) {
		*ptr = '_';
	}
	ptr=target;
	while ((ptr=strchr(ptr, '='))) {
		*ptr = '-';
	}
}

void mac64_to_b64(char *target)
{
	char *ptr;

	ptr=target;
	while ((ptr=strchr(ptr, '.'))) {
		*ptr = '+';
	}
	ptr=target;
	while ((ptr=strchr(ptr, '_'))) {
		*ptr = '/';
	}
	ptr=target;
	while ((ptr=strchr(ptr, '-'))) {
		*ptr = '=';
	}
}

/* BASE64-decoder */
int yphoto_b64_decode(char *target, int tlen, const char *source, int slen)
{
	const char *inp;
	char *outp;
	char *outend;
	const char *inend;
	int bits=0;
	int working=0;
	char *i;

	if (slen==0) slen = strlen(source);
	outp = target;
	inp = source;
	outend = target+tlen;
	inend = source+slen;
	
	for(;outp<outend && inp<inend;inp++) {
		if ( ((char)*inp) == '\n') {
			continue;
		}

		i = strchr(yphoto_b64s,(char)*inp);
		if (i == NULL) {
			return(-1);
		}

		if ((*i == '=') || (*i == '-')){ /* pad char */
			if ((working&0xFF) != 0) {
printf("PREVIEW NOT ENDED PROPERLY!!!, working = 0x%02x, bits: %d\n", working, bits);
				return(-1);
			}
			break;
		}

		bits += 6;
		working <<= 6;
		working |= (i-yphoto_b64s);
		
		if (bits >= 8) {
			*(outp++)=(char)((working&(0xFF<<(bits-8)))>>(bits-8));
			working &= (0xFF >> (8-(bits-8)));
			bits -= 8;
		}
	}

	if (outp == outend) {
		*(--outp)=0;
printf("PREVIEW BUFFER OVERFLOW!!!\n");
		return(-1);
	}

	*outp = 0;
	return(outp-target);
} /* yphoto_b64_decode */


/* BASE64-encoder helpfunction */
static int add_char(char *pos, char ch, int done, char *end)
{
	if (pos>=end) return(1);

	if (done) *pos = yphoto_b64s[64];
	else *pos = yphoto_b64s[(int)ch];

	return(0);
} /* add_char */


/* BASE64-encoder */
int yphoto_b64_encode(char *target, int tlen, const char *source, int slen)
{
	const char *inp;
	char *outp;
	char *outend;
	const char *inend;
	char *tmpbuf=NULL;
	int   done=0;
	char  enc;
	int   buf=0;

	if (slen==0) slen=strlen(source);
	inp=source;

	if (source==target) {
		tmpbuf = (char *)malloc(tlen);
		if (tmpbuf==NULL) {
			return(-1);
		}
		outp=tmpbuf;
	}
	else outp=target;

	outend = outp+tlen;
	inend = inp+slen;
	
	for(;(inp < inend) && !done;) {
		enc = *(inp++);
		buf = (enc << 4) & 0x30;
		enc = (enc >> 2) & 0x3F;

		if (add_char(outp++, enc, done, outend)) {
			if (target==source) {
				free(tmpbuf);
			}
			return(-1);
		}

		if (inp == inend) {
			enc = buf;
			buf = 0;
		}
		else {
			enc = buf | (((*inp) >> 4) & 0xF);
			buf = ((*inp) << 2) & 0x3C;
		}
		if (add_char(outp++, enc, done, outend))  {
			if (target==source) {
				free(tmpbuf);
			}
			return(-1);
		}

		if (inp == inend) {
			done=1;
		}
		else {
			inp++;
		}
		if (inp == inend) {
			enc = buf;
			buf = 0;
		}
		else {
			enc = buf | (((*inp) >> 6) & 0x03);
			buf = (*inp) & 0x3F;
		}
		if (add_char(outp++, enc, done, outend))  {
			if (target==source) {
				free(tmpbuf);
			}
			return(-1);
		}

		if (inp == inend) {
			done=1;
		}
		else {
			enc = buf;
			inp++;
		}
		if (add_char(outp++, enc, done, outend))  {
			if (target==source) {
				free(tmpbuf);
			}
			return(-1);
		}
		if (inp == inend) {
			done=1;
		}
	}

	if (outp<outend) *outp = 0;
	if (target==source)  {
		memcpy(target,tmpbuf,tlen);
		free(tmpbuf);
	}
	return(outp-target);
} /* yphoto_b64_encode */


/* yahoo's filekey is really a mac64 encoded md5 checksum.
 * Compute the md5 checksum, and then mac64 encode it.
 */
char *yphoto_genfilekey(char *file_path, GtkWidget *parent)
{
	int in_fd;
	unsigned char md5_result[16];
	md5_ctx_t md5_ctx;	
	char buffer[1024];
	int in_count;
	char *filekey;

	in_fd = open(file_path, O_RDONLY);
	if (in_fd < 0) {
		snprintf(buffer, sizeof(buffer)-1, _("Error opening input file %s for read.\n%s"),
			 file_path, 
			 strerror(errno));
		show_ok_dialog_p(parent, buffer);
		return NULL;
	}

	md5_init(&md5_ctx);
	while ( (in_count = read(in_fd, buffer, sizeof(buffer))) > 0 ) {
		md5_append(&md5_ctx, buffer, in_count);
	}
	md5_finish(&md5_ctx, md5_result);


	filekey = malloc(25);
	yphoto_b64_encode(filekey, 24, md5_result, 16);
	filekey[24] = 0x00;

	b64_to_mac64(filekey);

#if 0
{
  int   decoded_length;
  char *decoded_buffer, *p;
  char *encoded_buffer = strdup(filekey);
  decoded_length = strlen(encoded_buffer);
  decoded_buffer = malloc(decoded_length);
  mac64_to_b64(encoded_buffer);
  decoded_length = yphoto_b64_decode(decoded_buffer, decoded_length, encoded_buffer, decoded_length);
  decoded_buffer[decoded_length]=0;
  printf("filekey: %s\n", filekey);
  printf("decoded: "); for (p=decoded_buffer; *p ; p++) printf("%02x ", *p); printf("\n");
 }
#endif

	return(filekey);
} /* yphoto_genfilekey */


gchar *getfilenamefrompath(gchar *path)
{
	gchar *new;
	
	if (path) {
		// find the last slash
		new = strrchr(path, '/');
		return (gchar *)(++new);
	}
	return NULL;
}

PHOTO_INFO *find_info_by_filekey(PM_SESSION *pm_sess, char *filekey)
{
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info = NULL;
	GList *element;

	if (!filekey) {
		return(NULL);
	}

	for (element=photoshare->picturelist; element; element=element->next){
		photo_info = (PHOTO_INFO *)element->data;
		if (!strcmp(filekey, photo_info->filekey)) {
			break;
		}
		photo_info = NULL;
	}

	return(photo_info);
}

gboolean is_valid_photo_info(PM_SESSION *pm_sess, PHOTO_INFO *photo_info)
{
	PHOTOSHARE *photoshare;
	GList *element;

	if (g_list_index(pm_list, pm_sess) < 0) {
		/* pm_sess not in pm_linked list, so can't be valid */
		return(0);
	}

	photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	if (photoshare) {
		for (element=photoshare->picturelist; element; element=element->next){
			if (element->data == photo_info) {
				return(1);
			}
		}
	}

	return(0);
}

void set_counterlabel(PM_SESSION *pm_sess)
{
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	gchar tmp[10];

	if (photoshare->picturecount) {
		snprintf((gchar *)&tmp, sizeof(tmp), "%i/%i",
			 g_list_position(photoshare->picturelist, photoshare->current_thumb) + 1,
			 photoshare->picturecount);
	}
	else {	
		tmp[0] = 0;
	}
	gtk_label_set_text(GTK_LABEL(photoshare->counterlabel), (gchar *)&tmp);
}

/* The following taken almost verbatim from the devhelp GdkPixbuf page...
 *
 * For now, we're ignoring the alpha channel.
 */
void yphoto_draw_dot(GdkPixbuf *pixbuf, int x, int y, guchar red, guchar green, guchar blue)
{
	int width;
	int height;
	int n_channels;
	int row_stride;
	guchar *pixels, *p;

	width  = gdk_pixbuf_get_width(pixbuf);
	height = gdk_pixbuf_get_height(pixbuf);
	n_channels = gdk_pixbuf_get_n_channels(pixbuf);
	row_stride = gdk_pixbuf_get_rowstride(pixbuf);

	if ((x < 0) || (x >= width)) {
		return;
	}

	if ((y < 0) || (y >= height)) {
		return;
	}

	pixels = gdk_pixbuf_get_pixels(pixbuf);
	p = pixels + y * row_stride + x * n_channels;
	p[0] = red;
	p[1] = green;
	p[2] = blue;
}


/*
 * size <  0 --> draw photo to fit the preview box.
 * size == 0 --> use the last drawn size. If no last drawn size, then default to -1 (preview size)
 * size >  0 --> draw photo at full actual size.
 */
void redraw_selected_icon_image(PM_SESSION *pm_sess, int size)
{
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info;
	GdkPixbuf *pixbuf = NULL;
	GError *error = NULL;
	int height;
	int width;
	int x, y;

	if (photoshare->current_thumb) {
		photo_info = (PHOTO_INFO *)photoshare->current_thumb->data;

		/* rotation sanity */
		while (photo_info->rotation < 0)    photo_info->rotation += 360;
		while (photo_info->rotation >= 360) photo_info->rotation -= 360;

		if (size == 0) {
			size = photo_info->last_size;
			if (size == 0) {
				size = -1;
			}
		}

		if ( size < 0) {
			if (photo_info->rotation % 180) {
				width  = photoshare->preview_image_frame->allocation.height;
				height = photoshare->preview_image_frame->allocation.width;
			}
			else {
				height = photoshare->preview_image_frame->allocation.height;
				width  = photoshare->preview_image_frame->allocation.width;
			}
			height -= 5;
			width  -= 5;
		}
		else {
			height = photo_info->height;
			width  = photo_info->width;
		}


		if (height > photo_info->height) height = photo_info->height;
		if (width  > photo_info->height) width  = photo_info->width;

		pixbuf = gdk_pixbuf_new_from_file_at_size(photo_info->filename, width,height, &error);
		if (!error) {
			if (photo_info->pointer_on) {
				int my_height;
				int my_width;
				int my_x;
				int my_y;

				my_height = gdk_pixbuf_get_height(pixbuf);
				my_width  = gdk_pixbuf_get_width(pixbuf);

				my_x = photo_info->pointer_x * (1.0 * my_width)  / photo_info->width;
				my_y = photo_info->pointer_y * (1.0 * my_height) / photo_info->height;

				for (x = my_x - 10; x <= my_x + 10; x++) {
					yphoto_draw_dot(pixbuf, x, my_y, 0, 0, 0);
				}
				for (y = my_y - 10; y <= my_y + 10; y++) {
					yphoto_draw_dot(pixbuf, my_x, y, 0, 0, 0);
				}
			}

			if (photo_info->rotation) {
				GdkPixbuf *new_pixbuf = gdk_pixbuf_rotate_simple(pixbuf, photo_info->rotation);
				if (new_pixbuf) {
					g_object_unref(pixbuf);
					pixbuf = new_pixbuf;
				}
			}

			x = (photoshare->preview_image_frame->allocation.width  - gdk_pixbuf_get_width(pixbuf))/2;
			y = (photoshare->preview_image_frame->allocation.height - gdk_pixbuf_get_height(pixbuf))/2;

			if (x < 0) {
				x = 0;
			}
			if (y < 0) {
				y = 0;
			}

			gtk_fixed_move(GTK_FIXED(photoshare->preview_image_canvas),
				       photoshare->preview_image_ebox,
				       x, y);

			gtk_image_set_from_pixbuf(GTK_IMAGE(photoshare->preview_image), pixbuf);
			gyachi_set_tooltip(photoshare->preview_image, photo_info->hint);

		}
		else {
			g_error_free(error);
		}
		g_object_unref(pixbuf);

		pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)xpm_black);
		gtk_image_set_from_pixbuf(GTK_IMAGE(photo_info->bar), pixbuf);
		g_object_unref(pixbuf);

		photo_info->last_size = size;
	}
	else {
		gtk_image_clear(GTK_IMAGE(photoshare->preview_image));
		gyachi_set_tooltip(photoshare->preview_image, NULL);
	}

	set_counterlabel(pm_sess);
}


void free_photo_item(PM_SESSION *pm_sess, GList *element)
{
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info = (PHOTO_INFO *)element->data;

	if (photoshare->current_thumb == element) {
		photoshare->current_thumb = NULL;
	}

	gtk_widget_destroy(photo_info->button);
	photoshare->picturecount--;
	if (photo_info->direction == RECEIVING) {
		char *old_filename = malloc(strlen(photo_info->filename)+2);

		strcpy(old_filename, photo_info->filename);
		strcat(old_filename, "~");
		unlink(photo_info->filename);
		unlink(old_filename);
		free(old_filename);
	}

	if (photo_info->filename) {
		free(photo_info->filename);
	}

	if (photo_info->extn) {
		free(photo_info->extn);
	}

	if (photo_info->filekey) {
		free(photo_info->filekey);
	}

	if (photo_info->hint) {
		free(photo_info->hint);
	}

	if (photo_info->preview) {
		free(photo_info->preview);
	}

	if (photo_info->file_token) {
		free(photo_info->file_token);
	}

	free(photo_info);
}

void remove_photo(PM_SESSION *pm_sess, GList *current_element)
{
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *buddy_pic_info;
	GList *next_element;
	GList *buddy_element;
	GList *	current_thumb = photoshare->current_thumb;

	if (!current_element) {
		return;
	}

	next_element = current_element->next;

	/* check to see if the picture being removed is the pic in the current
	 * buddy view. If it is, then we need to invalidate it by clearing/freeing
	 * the remembered buddy_filekey.
	 * Also, clear the image, and tooltip.
	 */
	buddy_pic_info = find_info_by_filekey(pm_sess, photoshare->buddy_filekey);
	if (buddy_pic_info) {
		buddy_element = buddy_pic_info->my_element;
		if (buddy_element == current_element) {
			if (photoshare->buddy_filekey) {
				free(photoshare->buddy_filekey);
				photoshare->buddy_filekey = NULL;

				gtk_image_clear(GTK_IMAGE(photoshare->friend_view_image));
				gyachi_set_tooltip(photoshare->friend_view_image, NULL);
			}
		}
	}

	/* If the removed photo happens to be the current thumb being displayed,
	 * we will need to go to the next one. If there is no next one, then we need
	 * to go to the previous one.
	 */
	if (!next_element) {
		next_element = current_element->prev;
	}

	free_photo_item(pm_sess, current_element);
	photoshare->picturelist = g_list_delete_link(photoshare->picturelist, current_element);

	if (current_thumb == current_element) {
		photoshare->current_thumb = next_element;
		redraw_selected_icon_image(pm_sess, 0);
	}
	else {
		set_counterlabel(pm_sess);
	}
}

void free_photoshare(PM_SESSION *pm_sess)
{
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	GList *element;

	if (photoshare->buddy_filekey) {
		free(photoshare->buddy_filekey);
	}

	if (photoshare->picturelist) {
		while ((element=photoshare->picturelist)) {
			remove_photo(pm_sess, element);
		}
		g_list_free(photoshare->picturelist);
	}

	free(pm_sess->photoshare);
	pm_sess->photoshare = NULL;
}

void free_gslist(GSList **list)
{
	GSList *element;

	if (!list) return;

	for (element=*list; element; element=element->next) {
		g_free(element->data);
	}

	g_slist_free(*list);
	*list = NULL;
}

char *write_to_file(char *data, int length, char *dst_filename, GtkWidget *parent)
{
	char  *(pieces[4]);
	char *tmp_filename;
	int out_fd;
	char buffer[1024];
	int out_count;
	struct stat sbuf;

	pieces[0]=GYACH_CFG_DIR;
	pieces[1]="/photos";
	pieces[2]=NULL;

	tmp_filename = gyachi_filename(pieces);
	if (stat(tmp_filename, &sbuf )) {
                mkdir(tmp_filename, 0700 );
        }
	free(tmp_filename);

	pieces[1]="/photos/";
	pieces[2]=dst_filename;
	pieces[3]=NULL;
	tmp_filename=gyachi_filename(pieces);
	out_fd = creat(tmp_filename, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH);
	if (out_fd < 0) {
		snprintf(buffer, sizeof(buffer)-1, _("Error opening output file %s for create.\n%s"),
			 tmp_filename, 
			 strerror(errno));
		show_ok_dialog_p(parent, buffer);
		free(tmp_filename);
		tmp_filename = NULL;
		goto close_file;
	}

	out_count = write(out_fd, data, length);
	if (out_count < 0) {
		snprintf(buffer, sizeof(buffer)-1, _("Error writing file %s\n%s"),
			 tmp_filename, 
			 strerror(errno));
		show_ok_dialog_p(parent, buffer);
		free(tmp_filename);
		tmp_filename = NULL;
		goto close_file;
	}
	if (out_count != length) {
		snprintf(buffer, sizeof(buffer)-1, _("Error writing file %s\nExpected to write %d bytes, but only wrote %d bytes"),
			 tmp_filename, 
			 length, out_count);
		show_ok_dialog_p(parent, buffer);
		free(tmp_filename);
		tmp_filename = NULL;
		goto close_file;
	}

 close_file:
	close(out_fd);
	return(tmp_filename);
}

void save_to_file(char *src_filename, char *dst_filename, GtkWidget *parent)
{
	int in_fd;
	int out_fd;
	char buffer[1024];
	int in_count;
	int out_count;

	in_fd = open(src_filename, O_RDONLY);
	if (in_fd < 0) {
		snprintf(buffer, sizeof(buffer)-1, _("Error opening input file %s for read.\n%s"),
			 src_filename, 
			 strerror(errno));
		show_ok_dialog_p(parent, buffer);
		return;
	}

	out_fd = creat(dst_filename, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IRWXO);
	if (out_fd < 0) {
		snprintf(buffer, sizeof(buffer)-1, _("Error opening output file %s for create.\n%s"),
			 dst_filename, 
			 strerror(errno));
		show_ok_dialog_p(parent, buffer);
		goto close_inp_file;
	}

	while ( (in_count = read(in_fd, buffer, sizeof(buffer))) > 0 ) {
		out_count = write(out_fd, buffer, in_count);
		if (out_count < 0) {
			snprintf(buffer, sizeof(buffer)-1, _("Error writing file %s\n%s"),
				 dst_filename, 
				 strerror(errno));
			show_ok_dialog_p(parent, buffer);
			goto close_all_files;
		}
		if (out_count != in_count) {
			snprintf(buffer, sizeof(buffer)-1, _("Error writing file %s\nExpected to write %d bytes, but only wrote %d bytes"),
				 dst_filename, 
				 in_count, out_count);
			show_ok_dialog_p(parent, buffer);
			goto close_all_files;
		}
	}

	if (in_count < 0) {
		snprintf(buffer, sizeof(buffer)-1, _("Error reading file %s\n%s"),
			 src_filename, 
			 strerror(errno));
		show_ok_dialog_p(parent, buffer);
		goto close_all_files;
	}

 close_all_files:
	close(out_fd);
 close_inp_file:
	close(in_fd);
	return;
}


void on_mainwindow_destroy(GtkWidget *widget, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;

	/* MUST GO THRU AND FREE ANY PRIVATE STORAGE RELATED TO THE SESSION */
	if (pm_sess->photoshare) {
		free_photoshare(pm_sess);
	}

	ymsg_yphoto_close(ymsg_sess, pm_sess->pm_user);
}

void refresh_buddy_pic(PM_SESSION *pm_sess)
{
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *buddy_pic_info;
	PHOTO_INFO *photo_info;
	GtkImage   *thumbimage;

	buddy_pic_info = find_info_by_filekey(pm_sess, photoshare->buddy_filekey);
	if (!buddy_pic_info) {
		return;
	}

	thumbimage = g_object_get_data(G_OBJECT(buddy_pic_info->button), "thumbimage");
	if (thumbimage) {
		GdkPixbuf* pixbuf;

		pixbuf = gtk_image_get_pixbuf(thumbimage);
		if (pixbuf) {
			gtk_image_set_from_pixbuf((GtkImage *)photoshare->friend_view_image, pixbuf);
			gyachi_set_tooltip(photoshare->friend_view_image, buddy_pic_info->hint);
		}			
	}

	/* if "friend view" button is clicked in, then we need to update
	 * our preview image to what the buddy is viewing.
	 */
	if (photoshare->friendview_button_state) {
		photo_info = (PHOTO_INFO *)photoshare->current_thumb->data;
		gtk_image_clear(GTK_IMAGE(photo_info->bar));
		photoshare->current_thumb = buddy_pic_info->my_element;
		redraw_selected_icon_image(pm_sess, 0);
	}
}

void on_button_friendsview_clicked(GtkWidget *widget, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *buddy_pic_info;
	PHOTO_INFO *photo_info = photoshare->current_thumb?(PHOTO_INFO *)photoshare->current_thumb->data: NULL;

	photoshare->friendview_button_state = gtk_toggle_button_get_active((GtkToggleButton *)widget);
	if (photoshare->friendview_button_state) {
		buddy_pic_info = find_info_by_filekey(pm_sess, photoshare->buddy_filekey);
		if (buddy_pic_info) {
			if (photo_info) {
				gtk_image_clear(GTK_IMAGE(photo_info->bar));
			}
			photoshare->current_thumb = buddy_pic_info->my_element;
			redraw_selected_icon_image(pm_sess, 0);
		}
	}
}

void on_button_close_clicked(GtkWidget *widget, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	GtkWidget  *pm_session = pm_sess->pm_window;
	GtkWidget  *dlmanager;

	if (photoshare) {
		gtk_widget_destroy(photoshare->vbox);
	}

	/* close the d/l manager here */
	dlmanager = g_object_get_data(G_OBJECT(pm_session), "dlmanager");
	if (dlmanager) {
		gtk_widget_hide(dlmanager);
	}
}

void on_button_add_clicked(GtkWidget *widget, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;
	GtkWindow *parent = (GtkWindow *)pm_sess->pm_notebook->window;
	GtkWidget *dialog;
	gchar     *path;
	int       rv;
	GSList    *uri_list, *element;

	dialog = gtk_file_chooser_dialog_new ("Select File(s)",
					      parent,
					      GTK_FILE_CHOOSER_ACTION_OPEN,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					      NULL);

	gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), photoshare_dir);
	gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(dialog), TRUE);

	/* Create the selector */
	rv = gtk_dialog_run(GTK_DIALOG (dialog));
	if (rv == GTK_RESPONSE_ACCEPT) {
		uri_list = gtk_file_chooser_get_uris(GTK_FILE_CHOOSER(dialog));
		for (element=uri_list; element; element=element->next) {
			path = g_filename_from_uri(element->data, NULL, NULL);
			if (path) {
				yphoto_add_file(pm_sess, path, NULL, NULL);
				g_free(path);
			}
		}		
		free_gslist(&uri_list);
		redraw_selected_icon_image(pm_sess, 0);

		if (strcmp(photoshare_dir, gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(dialog)))) {
			free(photoshare_dir);
			photoshare_dir = strdup(gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(dialog)));
			write_config();
		}
	}

	gtk_widget_destroy(dialog);
}

void on_yphoto_remove_activate(GtkWidget *widget, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info;
	GList *current_link;
	GList *current_thumb = photoshare->current_thumb;
	GtkWidget *tmp;

	/* look for the link in the menu_item's "photo_link" data object.
	 * It need not be set for the Tools menu popup, and it need not be
	 * set for the preview right-mouse-click, but *MUST* be set for the
	 * popup on the icon bar images (otherwise we'll end up removing
	 * the image in the preview window!)
	 */
	current_link = 0;
	tmp = g_object_get_data(G_OBJECT(widget), "photo_info");
	if (tmp) {
		photo_info = (PHOTO_INFO *)tmp;
		current_link = photo_info->my_element;
	}

	if (!current_link) {
		current_link = photoshare->current_thumb;
	}
	if (!current_link) {
		return;
	}

	photo_info = (PHOTO_INFO *)current_link->data;
	ymsg_yphoto_remove(ymsg_sess, pm_sess->pm_user, photo_info->filekey);
	
	remove_photo(pm_sess, current_link);

	/* if we ended up deleting the current thumb, then we need to redisplay the
	 * new current thumb, and also notify our buddy of our new thumb's key
	 */
	if (current_thumb != photoshare->current_thumb) {
		if (photoshare->current_thumb) {
			photo_info = (PHOTO_INFO *)photoshare->current_thumb->data;
			ymsg_yphoto_key(ymsg_sess, pm_sess->pm_user, photo_info->filekey, photoshare->picture_sent_count);
			photoshare->picture_sent_count++;
		}
	}
}

void on_preview_pointer_activate(GtkWidget *widget, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info = photoshare->current_thumb?(PHOTO_INFO *)photoshare->current_thumb->data: NULL;

	if (!photo_info) {
		return;
	}

	photo_info->pointer_on = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget));
	if (!photo_info->pointer_on) {
		ymsg_yphoto_pointer(ymsg_sess, pm_sess->pm_user,
				    photo_info->filekey,
				    photo_info->pointer_x,
				    photo_info->pointer_y,
				    photo_info->pointer_on);

	}

	redraw_selected_icon_image(pm_sess, 0);
}

void on_preview_bestfit_activate(GtkWidget *widget, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;

	redraw_selected_icon_image(pm_sess, -1);
}

void on_preview_fullsize_activate(GtkWidget *widget, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;

	redraw_selected_icon_image(pm_sess, 1);
}

void on_preview_clockwise_activate(GtkWidget *widget, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info = photoshare->current_thumb?(PHOTO_INFO *)photoshare->current_thumb->data: NULL;
	GtkImage   *thumbimage;
	GtkImage   *tmp;

	/* for thumb preview menu, we pass the photo_info by way of a g_object_data in the button.
	 * for preview menu, and for the [Save] button, we use the current thumb
	 */
	tmp = g_object_get_data(G_OBJECT(widget), "photo_info");
	if (tmp) {
		photo_info = (PHOTO_INFO *)tmp;
	}

	if (!photo_info) {
		return;
	}
	photo_info->rotation -= 90;

	redraw_selected_icon_image(pm_sess, 0);

	thumbimage = g_object_get_data(G_OBJECT(photo_info->button), "thumbimage");
	if (thumbimage) {
		GdkPixbuf* pixbuf;

		pixbuf = gtk_image_get_pixbuf(thumbimage);
		if (pixbuf) {
			GdkPixbuf *new_pixbuf = gdk_pixbuf_rotate_simple(pixbuf, 270);
			if (new_pixbuf) {
				gtk_image_set_from_pixbuf(thumbimage, new_pixbuf);
			}			
		}
	}	
}

void on_preview_counterclockwise_activate(GtkWidget *widget, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info = photoshare->current_thumb?(PHOTO_INFO *)photoshare->current_thumb->data: NULL;
	GtkImage   *thumbimage;
	GtkImage   *tmp;

	/* for thumb preview menu, we pass the photo_info by way of a g_object_data in the button.
	 * for preview menu, and for the [Save] button, we use the current thumb
	 */
	tmp = g_object_get_data(G_OBJECT(widget), "photo_info");
	if (tmp) {
		photo_info = (PHOTO_INFO *)tmp;
	}

	if (!photo_info) {
		return;
	}
	photo_info->rotation += 90;

	redraw_selected_icon_image(pm_sess, 0);

	thumbimage = g_object_get_data(G_OBJECT(photo_info->button), "thumbimage");
	if (thumbimage) {
		GdkPixbuf* pixbuf;

		pixbuf = gtk_image_get_pixbuf(thumbimage);
		if (pixbuf) {
			GdkPixbuf *new_pixbuf = gdk_pixbuf_rotate_simple(pixbuf, 90);
			if (new_pixbuf) {
				gtk_image_set_from_pixbuf(thumbimage, new_pixbuf);
			}			
		}
	}
}

void on_preview_save_activate(GtkWidget *widget, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info = photoshare->current_thumb?(PHOTO_INFO *)photoshare->current_thumb->data: NULL;
	GtkWindow *parent = (GtkWindow *)pm_sess->pm_notebook->window;
	GtkWidget *dialog;
	GtkWidget *tmp;
	int       rv;
	char *filename;

	/* for thumb preview menu, we pass the photo_info by way of a g_object_data in the button.
	 * for preview menu, and for the [Save] button, we use the current thumb
	 */
	tmp = g_object_get_data(G_OBJECT(widget), "photo_info");
	if (tmp) {
		photo_info = (PHOTO_INFO *)tmp;
	}

	if (!photo_info) {
		return;
	}

	dialog = gtk_file_chooser_dialog_new ("Save file",
					      parent,
					      GTK_FILE_CHOOSER_ACTION_SAVE,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
					      NULL);

	gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog), TRUE);
	gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), photoshare_dir);
	gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), photo_info->hint);
	gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(dialog), FALSE);

	/* Create the selector */
	rv = gtk_dialog_run(GTK_DIALOG (dialog));
	if (rv == GTK_RESPONSE_ACCEPT) {
		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
		save_to_file(photo_info->filename, filename, (GtkWidget *)parent);
		g_free (filename);

		if (strcmp(photoshare_dir, gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(dialog)))) {
			free(photoshare_dir);
			photoshare_dir = strdup(gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(dialog)));
			write_config();
		}
	}

	gtk_widget_destroy(dialog);
}

GtkWidget *create_preview_popup_menu(gpointer pm_sess)
{
	GtkWidget *preview_menu;
	GtkWidget *bestfit_item;
	GtkWidget *fullsize_item;
	GtkWidget *clockwise_item;
	GtkWidget *counterclockwise_item;
	GtkWidget *pointer_item;
	GtkWidget *remove_item;
	GtkWidget *save_item;

	/* This is the popup menu for the preview image */
	preview_menu = gtk_menu_new();
	bestfit_item = gtk_image_menu_item_new_with_label(_("Bestfit"));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(bestfit_item), 
				      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_ZOOM_FIT,
									  GTK_ICON_SIZE_MENU)));
	gtk_container_add(GTK_CONTAINER(preview_menu), bestfit_item);
	g_signal_connect(G_OBJECT(bestfit_item), "activate",
			 G_CALLBACK(on_preview_bestfit_activate),
			 pm_sess);

	fullsize_item = gtk_image_menu_item_new_with_label(_("Fullsize"));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(fullsize_item), 
				      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_ZOOM_OUT,
									  GTK_ICON_SIZE_MENU)));
	gtk_container_add(GTK_CONTAINER(preview_menu), fullsize_item);
	g_signal_connect(G_OBJECT(fullsize_item), "activate",
			 G_CALLBACK(on_preview_fullsize_activate),
			 pm_sess);

	/* separator */
	gtk_container_add(GTK_CONTAINER(preview_menu), gtk_menu_item_new());

	clockwise_item = gtk_image_menu_item_new_with_label(_("Rotate +90"));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(clockwise_item), 
				      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_REDO,
									  GTK_ICON_SIZE_MENU)));
	gtk_container_add(GTK_CONTAINER(preview_menu), clockwise_item);
	g_signal_connect(G_OBJECT(clockwise_item), "activate",
			 G_CALLBACK(on_preview_clockwise_activate),
			 pm_sess);

	counterclockwise_item = gtk_image_menu_item_new_with_label(_("Rotate -90"));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(counterclockwise_item), 
				      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_UNDO,
									  GTK_ICON_SIZE_MENU)));
	gtk_container_add(GTK_CONTAINER(preview_menu), counterclockwise_item);
	g_signal_connect(G_OBJECT(counterclockwise_item), "activate",
			 G_CALLBACK(on_preview_counterclockwise_activate),
			 pm_sess);

	/* separator */
	gtk_container_add(GTK_CONTAINER(preview_menu), gtk_menu_item_new());

	pointer_item = gtk_check_menu_item_new_with_label(_(" Pointer"));
	g_object_set_data(G_OBJECT(preview_menu), "pointer_item", pointer_item);
	gtk_container_add(GTK_CONTAINER(preview_menu), pointer_item);
	g_signal_connect(G_OBJECT(pointer_item), "toggled",
			 G_CALLBACK(on_preview_pointer_activate),
			 pm_sess);

	/* separator */
	gtk_container_add(GTK_CONTAINER(preview_menu), gtk_menu_item_new());

	remove_item = gtk_image_menu_item_new_with_label(_("Remove"));
	g_object_set_data(G_OBJECT(preview_menu), "remove_item", remove_item);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(remove_item), 
				      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_REMOVE,
									  GTK_ICON_SIZE_MENU)));

	gtk_container_add(GTK_CONTAINER(preview_menu), remove_item);
	g_signal_connect(G_OBJECT(remove_item), "activate",
			 G_CALLBACK(on_yphoto_remove_activate),
			 pm_sess);

	/* separator */
	gtk_container_add(GTK_CONTAINER(preview_menu), gtk_menu_item_new());

	save_item = gtk_image_menu_item_new_with_label(_("Save"));
	g_object_set_data(G_OBJECT(preview_menu), "save_item", save_item);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(save_item), 
				      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_SAVE,
									  GTK_ICON_SIZE_MENU)));
	gtk_container_add(GTK_CONTAINER(preview_menu), save_item);
	g_signal_connect(G_OBJECT(save_item), "activate",
			 G_CALLBACK(on_preview_save_activate),
			 pm_sess);

	return(preview_menu);
}

GtkWidget *create_thumb_popup_menu(gpointer pm_sess, PHOTO_INFO *photo_info)
{
	GtkWidget *preview_menu;
	GtkWidget *clockwise_item;
	GtkWidget *counterclockwise_item;
	GtkWidget *remove_item;
	GtkWidget *save_item;

	/* This is the popup menu for the thumb image */
	preview_menu = gtk_menu_new();
	clockwise_item = gtk_image_menu_item_new_with_label(_("Rotate +90"));
	g_object_set_data(G_OBJECT(clockwise_item), "photo_info", photo_info);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(clockwise_item), 
				      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_REDO,
									  GTK_ICON_SIZE_MENU)));
	gtk_container_add(GTK_CONTAINER(preview_menu), clockwise_item);
	g_signal_connect(G_OBJECT(clockwise_item), "activate",
			 G_CALLBACK(on_preview_clockwise_activate),
			 pm_sess);

	counterclockwise_item = gtk_image_menu_item_new_with_label(_("Rotate -90"));
	g_object_set_data(G_OBJECT(counterclockwise_item), "photo_info", photo_info);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(counterclockwise_item), 
				      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_UNDO,
									  GTK_ICON_SIZE_MENU)));
	gtk_container_add(GTK_CONTAINER(preview_menu), counterclockwise_item);
	g_signal_connect(G_OBJECT(counterclockwise_item), "activate",
			 G_CALLBACK(on_preview_counterclockwise_activate),
			 pm_sess);

	/* separator */
	gtk_container_add(GTK_CONTAINER(preview_menu), gtk_menu_item_new());

	remove_item = gtk_image_menu_item_new_with_label(_("Remove"));
	g_object_set_data(G_OBJECT(remove_item), "photo_info", photo_info);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(remove_item), 
				      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_REMOVE,
									  GTK_ICON_SIZE_MENU)));

	gtk_container_add(GTK_CONTAINER(preview_menu), remove_item);
	g_signal_connect(G_OBJECT(remove_item), "activate",
			 G_CALLBACK(on_yphoto_remove_activate),
			 pm_sess);

	/* separator */
	gtk_container_add(GTK_CONTAINER(preview_menu), gtk_menu_item_new());

	save_item = gtk_image_menu_item_new_with_label(_("Save"));
	g_object_set_data(G_OBJECT(save_item), "photo_info", photo_info);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(save_item), 
				      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_SAVE,
									  GTK_ICON_SIZE_MENU)));
	gtk_container_add(GTK_CONTAINER(preview_menu), save_item);
	g_signal_connect(G_OBJECT(save_item), "activate",
			 G_CALLBACK(on_preview_save_activate),
			 pm_sess);

	return(preview_menu);
}

void yphoto_unnormalze(PM_SESSION *pm_sess, int mouse_x, int mouse_y)
{
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info = photoshare->current_thumb?(PHOTO_INFO *)photoshare->current_thumb->data: NULL;
	GdkPixbuf *pixbuf;
	int height;
	int width;
	int image_x;
	int image_y;

	pixbuf = gtk_image_get_pixbuf(GTK_IMAGE(photoshare->preview_image));
	height = gdk_pixbuf_get_height(pixbuf);
	width  = gdk_pixbuf_get_width(pixbuf);


	switch (photo_info->rotation) {
	case 0:
		image_x = mouse_x * (1.0*photo_info->width)/width;
		image_y = mouse_y * (1.0*photo_info->height)/height;
		photo_info->pointer_x = image_x;
		photo_info->pointer_y = image_y;
		break;

	case 90:
		image_x = mouse_x * (1.0*photo_info->height)/width;
		image_y = mouse_y * (1.0*photo_info->width)/height;
		photo_info->pointer_x = photo_info->width  - image_y;
		photo_info->pointer_y = image_x;
		break;

	case 180:
		image_x = mouse_x * (1.0*photo_info->width)/width;
		image_y = mouse_y * (1.0*photo_info->height)/height;
		photo_info->pointer_x = photo_info->width  - image_x;
		photo_info->pointer_y = photo_info->height - image_y;
		break;

	case 270:
		image_x = mouse_x * (1.0*photo_info->height)/width;
		image_y = mouse_y * (1.0*photo_info->width)/height;
		photo_info->pointer_x = image_y;
		photo_info->pointer_y = photo_info->height - image_x;
		break;
	}
}

void on_preview_clicked(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info = photoshare->current_thumb?(PHOTO_INFO *)photoshare->current_thumb->data: NULL;
	GtkWidget  *pointer_item;
	GtkWidget  *remove_item;

	if (!photo_info) {
		/* ignore clicks on the preview image box, if there is no preview selected */
		return;
	}

	/* Right mouse button clicks: Popup menu, Left mouse switchs between full/best fit */
	if (event->type != GDK_BUTTON_PRESS) {
		return;
	}
	switch (event->button) {
	case 1:
		if (photo_info->pointer_on && photo_info->in_image) {
			/* user click is to position the pointer... */
			/* normalize to real image coordinates, and then 
			 * redraw. The redraw code will take care of
			 * scaling and rotation...
			 */
			yphoto_unnormalze(pm_sess, event->x, event->y);
			redraw_selected_icon_image(pm_sess, 0);

			ymsg_yphoto_pointer(ymsg_sess, pm_sess->pm_user,
					    photo_info->filekey,
					    photo_info->pointer_x,
					    photo_info->pointer_y,
					    photo_info->pointer_on);
		}
		else {
			redraw_selected_icon_image(pm_sess, -photo_info->last_size);
		}
		break;
		
	case 3:
		if (!photoshare->popup_menu) {
			photoshare->popup_menu = create_preview_popup_menu(pm_sess);
		}

		pointer_item = g_object_get_data(G_OBJECT(photoshare->popup_menu), "pointer_item");
		if (pointer_item) {
			gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pointer_item), photo_info->pointer_on);
			gtk_widget_set_sensitive(pointer_item, photo_info->pointer_supported);
		}

		remove_item = g_object_get_data(G_OBJECT(photoshare->popup_menu), "remove_item");
		if (remove_item) {
			g_object_set_data(G_OBJECT(remove_item), "photo_link", photoshare->current_thumb);
		}

		gtk_widget_show_all(GTK_WIDGET(photoshare->popup_menu));

		/* right clicked popup action window */
		gtk_menu_popup((GtkMenu *)photoshare->popup_menu, NULL, NULL, NULL, NULL, 1, 0);
		break;
	}
}

gboolean on_thumb_clicked(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info = photoshare->current_thumb?(PHOTO_INFO *)photoshare->current_thumb->data: NULL;

	/* Right mouse button clicks: Popup menu, Left mouse selects thumb image for display */
	if (event->type != GDK_BUTTON_RELEASE) {
		return TRUE;
	}

	switch (event->button) {
	case 1:	
		if (photo_info) {
			gtk_image_clear(GTK_IMAGE(photo_info->bar));
		}

		photo_info = g_object_get_data(G_OBJECT(widget), "photo_info");
		if (photo_info) {
			photoshare->current_thumb = photo_info->my_element;
			ymsg_yphoto_key(ymsg_sess,  pm_sess->pm_user, photo_info->filekey, photoshare->picture_sent_count);
			photoshare->picture_sent_count++;
			redraw_selected_icon_image(pm_sess, 0);
		}

		/* FIXME --> VERIFY --> when we click an image, it automatically sets [Friend View] to false... */
		if (photoshare->friendview_button_state) {
			gtk_toggle_button_set_active((GtkToggleButton *)photoshare->button_friendsview, FALSE);
		}
		break;

	case 3:
		photo_info = g_object_get_data(G_OBJECT(widget), "photo_info");
		if (!photo_info) {
			break;
		}

		if (!photo_info->popup_menu) {
			photo_info->popup_menu = create_thumb_popup_menu(pm_sess, photo_info);
			gtk_widget_show_all(GTK_WIDGET(photo_info->popup_menu));
		}


		/* right clicked popup action window */
		gtk_menu_popup((GtkMenu *)photo_info->popup_menu, NULL, NULL, NULL, NULL, 1, 0);
		break;

	}
	return TRUE;
}


typedef enum { H_BAR=1, V_BAR } SCROLLBAR;
void yphoto_adjustment_value_changed(GtkAdjustment *adj, PM_SESSION *pm_sess, SCROLLBAR s_bar)
{
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info;
	GtkWidget *scrollbox;

	if (photoshare) {
		if (photoshare->current_thumb) {
			photo_info = (PHOTO_INFO *)photoshare->current_thumb->data;

			if (photo_info->last_size > 0) {
				/* full size. Save scroll adjustments */

				scrollbox = photoshare->scrollbox;

				if (s_bar == H_BAR) {
					photo_info->h_adj = gtk_adjustment_get_value(adj);
				}
				else {
					photo_info->v_adj = gtk_adjustment_get_value(adj);
				}
			}
		}
	}
}

void yphoto_hadjustment_value_changed(GtkAdjustment *adjustment, PM_SESSION *pm_sess)
{
	yphoto_adjustment_value_changed(adjustment, pm_sess, H_BAR);
}

void yphoto_vadjustment_value_changed(GtkAdjustment *adjustment, PM_SESSION *pm_sess)
{
	yphoto_adjustment_value_changed(adjustment, pm_sess, V_BAR);
}

void yphoto_adjustment_changed(GtkAdjustment *adj, PM_SESSION *pm_sess, SCROLLBAR s_bar)
{
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info;
	GtkWidget *scrollbox;

	if (photoshare) {
		if (photoshare->current_thumb) {
			photo_info = (PHOTO_INFO *)photoshare->current_thumb->data;

			if (photo_info->last_size > 0) {
				/* full size. Save scroll adjustments */

				scrollbox = photoshare->scrollbox;

				if (s_bar == H_BAR) {
					gtk_adjustment_set_value(adj, photo_info->h_adj);
				}
				else {
					gtk_adjustment_set_value(adj, photo_info->v_adj);
				}
			}
		}
	}
}

void yphoto_hadjustment_changed(GtkAdjustment *adjustment, PM_SESSION *pm_sess)
{
	yphoto_adjustment_changed(adjustment, pm_sess, H_BAR);
}

void yphoto_vadjustment_changed(GtkAdjustment *adjustment, PM_SESSION *pm_sess)
{
	yphoto_adjustment_changed(adjustment, pm_sess, V_BAR);
}

gboolean on_image_enter(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info = photoshare->current_thumb?(PHOTO_INFO *)photoshare->current_thumb->data: NULL;

	if (!photo_info) {
		return FALSE;
	}

	photo_info->in_image = 1;

	if (!photo_info->pointer_on) {
		return FALSE;
	}

	gdk_window_set_cursor(widget->window, yphoto_cursor_tcross);

	return TRUE;
}

gboolean on_image_leave(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info = photoshare->current_thumb?(PHOTO_INFO *)photoshare->current_thumb->data: NULL;

	if (!photo_info) {
		return FALSE;
	}

	photo_info->in_image = 0;

	gdk_window_set_cursor(widget->window, NULL);

	if (!photo_info->pointer_on) {
		return FALSE;
	}

	return TRUE;
}

/* create jpeg from preview-pixbuf */
int yphoto_genpreview(GdkPixbuf *pixbuf, char **jpegbuf_data, char *filekey)
{
	JSAMPLE *image_buffer;
	struct jpeg_compress_struct cinfo;
	struct jpeg_error_mgr jerr;
	char *filename;
	FILE *outfile;
	JSAMPROW row_pointer[1];
	int row_stride;
	int jpegbuf_len;

	image_buffer = (JSAMPLE*)gdk_pixbuf_get_pixels(pixbuf);
	
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_compress(&cinfo);
	filename = malloc(strlen(filekey)+13);
	strcpy(filename, "/tmp/gyachi_");
	strcat(filename, filekey);

	outfile = fopen(filename, "w+");
	jpeg_stdio_dest(&cinfo, outfile);
	
	cinfo.image_width = gdk_pixbuf_get_width(pixbuf);
	cinfo.image_height = gdk_pixbuf_get_height(pixbuf);
	cinfo.input_components = 3;
	cinfo.in_color_space = JCS_RGB;
	jpeg_set_defaults(&cinfo);

	jpeg_start_compress(&cinfo, TRUE);
	row_stride = gdk_pixbuf_get_rowstride(pixbuf);

	while(cinfo.next_scanline < cinfo.image_height) {
		row_pointer[0] = &image_buffer[cinfo.next_scanline * row_stride];
		jpeg_write_scanlines(&cinfo, row_pointer, 1);
	}

	jpeg_finish_compress(&cinfo);

	jpegbuf_len = ftell(outfile);
	*jpegbuf_data = malloc(jpegbuf_len+1);
	rewind(outfile);
	fread(*jpegbuf_data, jpegbuf_len, 1, outfile);
	(*jpegbuf_data)[jpegbuf_len] = 0;
	fclose(outfile);

	unlink(filename);
	free(filename);
	jpeg_destroy_compress(&cinfo);
	return(jpegbuf_len);
} /* yphoto_genpreview */

void start_next_sendfile()
{
	SENDFILE_QUEUE_INFO *sendfile_queue_info;
	PHOTO_INFO *photo_info = NULL;
	PM_SESSION *pm_sess;
	PHOTOSHARE *photoshare;

	if (g_list_length(sendfile_queue)) {
		sendfile_queue_info = (SENDFILE_QUEUE_INFO *)g_list_first(sendfile_queue)->data;
		photo_info = sendfile_queue_info->photo_info;
		pm_sess    = sendfile_queue_info->pm_sess;
		photoshare = (PHOTOSHARE *)pm_sess->photoshare;

		ymsg_yphoto_prev(ymsg_sess,
				 pm_sess->pm_user,
				 photo_info->filekey, photo_info->extn,
				 photo_info->hint,
				 photo_info->preview);
		ymsg_yphoto_key(ymsg_sess, pm_sess->pm_user, photo_info->filekey, photoshare->picture_sent_count);
		photoshare->picture_sent_count++;
		if (photoshare->picture_sent_count == 1) {
			ymsg_yphoto_key(ymsg_sess, pm_sess->pm_user, photo_info->filekey, photoshare->picture_sent_count);
			photoshare->picture_sent_count++;
		}

		free(photo_info->preview);
		photo_info->preview=NULL;
	}
}

// Callback-functions
void yphoto_add_file(PM_SESSION *pm_sess, char *path, char *hint, char *the_filekey)
{
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	PHOTO_INFO *photo_info = NULL;
	GdkPixbuf *pixbuf = NULL;
	GtkWidget *parent;
	GError *error = NULL;
	char *filekey;
	char *extn, *ptr;
	char *preview;
	int   prevlen;
	GtkWidget *button;
	GtkWidget *ebox;
	GtkWidget *vbox;
	GtkWidget *thumbimage;
	GtkWidget *bar;
	struct stat stat_bf;
	GdkPixbufFormat *fileinfo;
	int        height;
	int        width;

	if (stat(path, &stat_bf)) {
		/* the file ought to stat. We need the filesize.
		 * If the file doesn't stat, then ignore it...
		 */
		return;
	}

	extn = NULL;
	fileinfo = gdk_pixbuf_get_file_info(path, &width, &height);
	if (!fileinfo) {
		/* the file format is not recognized.
		 * ignore it...
		 */
		return;
	}

	pixbuf = gdk_pixbuf_new_from_file_at_size(path, 48, 48, &error);
	if (!error) {
		if (!hint) {
			hint = getfilenamefrompath(path);
		}
		if (the_filekey) {
			filekey = strdup(the_filekey);
			ptr=filekey+24;
			if (strlen(filekey) > 24) {
				if (*ptr == '.') {
					extn = strdup(ptr);
					*ptr = 0;
				}
			}

			/* don't allow dups into the photolist */
			photo_info = find_info_by_filekey(pm_sess, filekey);
			if (photo_info) {
				free(filekey);
				return;
			}

			preview = 0;
			prevlen = 0;
		}
		else {
			char *jpgdata;
			int jpgdatalen;
			char **extns;

			parent = pm_sess->pm_notebook->window;
			filekey = yphoto_genfilekey(path, parent);

			/* don't allow dups into the photolist */
			photo_info = find_info_by_filekey(pm_sess, filekey);
			if (photo_info) {
				free(filekey);
				return;
			}

			extns = gdk_pixbuf_format_get_extensions(fileinfo);
			if (extns) {
				if (!strcmp(*extns, "jpg") && !strcmp(*extns, "jpe") && !strcmp(*extns, "jpeg")) {
					extn = malloc(strlen(*extns) + 2);
					*extn = '.';
					strcpy(extn+1, *extns);
				}
				g_strfreev(extns);
			}

			jpgdatalen = yphoto_genpreview(pixbuf, &jpgdata, filekey);

			prevlen = jpgdatalen*2;
			preview = malloc(prevlen);
			prevlen = yphoto_b64_encode(preview, prevlen, jpgdata, jpgdatalen);
			free(jpgdata);

			preview[prevlen] = '\0';
		}

		if (photoshare->current_thumb) {
			photo_info = (PHOTO_INFO *)photoshare->current_thumb->data;
			gtk_image_clear(GTK_IMAGE(photo_info->bar));
		}

		photo_info = malloc(sizeof(PHOTO_INFO));
		photo_info->filename     = strdup(path);
		photo_info->extn         = extn;
		photo_info->hint         = strdup(hint);
		photo_info->filekey      = filekey;
		photo_info->file_token   = NULL;
		photo_info->filesize     = stat_bf.st_size;
		photo_info->height       = height;
		photo_info->width        = width;
		photo_info->rotation     = 0;
		photo_info->last_size    = -1;
		photo_info->direction    = the_filekey?RECEIVING:SENDING;
		photo_info->my_element   = NULL;
		photo_info->button       = NULL;
		photo_info->bar          = NULL;
		photo_info->preview      = preview;
		photo_info->popup_menu   = NULL;
		photo_info->pointer_on   = 0;
		photo_info->pointer_x    = 0;
		photo_info->pointer_y    = 0;
		photo_info->in_image     = 0;

		photo_info->pointer_supported = 1;
		if ((gdk_pixbuf_get_n_channels (pixbuf) < 3) ||
		    (gdk_pixbuf_get_n_channels (pixbuf) > 4) ||
		    (gdk_pixbuf_get_colorspace (pixbuf) != GDK_COLORSPACE_RGB) ||
		    (gdk_pixbuf_get_bits_per_sample (pixbuf) != 8)) {
			photo_info->pointer_supported = 0;
		}
		
		photoshare->picturelist  = g_list_append(photoshare->picturelist, photo_info);
		photo_info->my_element   = g_list_find(photoshare->picturelist, photo_info);
		photoshare->picturecount++;
		photoshare->current_thumb = photo_info->my_element;

		if (photo_info->direction == SENDING) {

#if 0
			list_len = g_list_length(photoshare->picturelist);
			if (list_len == 0) {
				ymsg_yphoto_key(ymsg_sess,  pm_sess->pm_user, filekey, 0);
			}
#endif

#if 0

			ymsg_yphoto_prev(ymsg_sess, pm_sess->pm_user, photo_info->filekey, photo_info->hint, photo_info->preview);
			ymsg_yphoto_key(ymsg_sess, pm_sess->pm_user, photo_info->filekey, photoshare->picture_sent_count);
			photoshare->picture_sent_count++;
			free(photo_info->preview);
			photo_info->preview=NULL;
#else
			/* yahoo seems to have a glitch, where by if 2 file transfers start, at about the same
			 * time, then they will BOTH end up with the same "file token" (field 251).
			 * Yeah, BOTH. Yeah, *yuck*.
			 * For now, just queue up the entry. Check to see if we have an active runner. If not
			 * dequeue the top element. If so, nothing to do. When this file transfer completes,
			 * the completion code will also also dequeue the top element.
			 */
			SENDFILE_QUEUE_INFO *sendfile_queue_info;

			sendfile_queue_info = malloc(sizeof(SENDFILE_QUEUE_INFO));
			sendfile_queue_info->photo_info = photo_info;
			sendfile_queue_info->pm_sess    = pm_sess;
			
			sendfile_queue = g_list_append(sendfile_queue, sendfile_queue_info);
			if (g_list_length(sendfile_queue) == 1) {
				/* queue WAS empty. We just added the only element... */
				start_next_sendfile();
			}
#endif
		}

		/* vbox (button)
		 *  event_box (ebox)
		 *   vbox
		 *    image(thumbimage)
		 *    image(bar)
		 */
		button = gtk_vbox_new(FALSE, 0);
		ebox = gtk_event_box_new();
		gtk_container_add(GTK_CONTAINER(button), ebox);
		vbox = gtk_vbox_new(FALSE, 2);
		gtk_container_add(GTK_CONTAINER(ebox), vbox);

		thumbimage = gtk_image_new();
		gtk_image_set_from_pixbuf(GTK_IMAGE(thumbimage), pixbuf);
		gtk_widget_set_size_request(thumbimage, 48, 48);
		gtk_box_pack_start(GTK_BOX(vbox), thumbimage, FALSE, FALSE, 0);

		bar = gtk_image_new();
		gtk_widget_set_size_request(bar, 48, 5);
		gtk_box_pack_start(GTK_BOX(vbox), bar, FALSE, FALSE, 0);
		gyachi_set_tooltip(thumbimage, hint);
		gtk_box_pack_start(GTK_BOX(photoshare->hscrollbox), button, FALSE, FALSE, 0);
		g_object_set_data(G_OBJECT(button), "thumbimage", thumbimage);
		g_object_set_data(G_OBJECT(ebox),   "photo_info", photo_info);
		g_signal_connect(ebox, "button-release-event", G_CALLBACK(on_thumb_clicked), pm_sess);
		photo_info->button = button;	/* for button destruction   */
		photo_info->bar    = bar;	/* for highlighting the bar */

		gtk_widget_show_all(button);

		g_object_unref(pixbuf);
	}
	else {
		g_error_free(error);
	}
}

void on_drag_data_received (GtkWidget *widget, GdkDragContext *context1, gint x1, gint y1,
			    GtkSelectionData *selection_data, guint info1, guint32 time1,
			    gpointer data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)data;
	gchar **uriarray = NULL;
	gchar **uris;
	gchar *oneuri;
	gchar *path;
	
	if (selection_data->length > 0) {
		uriarray = gtk_selection_data_get_uris(selection_data);
		if (uriarray) {
			uris = uriarray;
			while((oneuri = *uris) != NULL) {
				path = g_filename_from_uri(oneuri, NULL, NULL);
				if (path) {
					yphoto_add_file(pm_sess, path, NULL, NULL);
					uris++;
					g_free(path);
				}
			}
 			g_strfreev(uriarray);
			redraw_selected_icon_image(pm_sess, 0);
		}
	}
}


/* Create the photosharing-gui */
/* This documents the right hand side gtk_paned_pack2 box (created in pmwindow.c)
 *
 *  vbox (photoshare_vbox)			photoshare->vbox
 *    event_box (eboxprev)
 *      vbox (preview_image_frame)		photoshare->preview_image_frame
 *        scrolled_window (scrollbox)		photoshare->scrollbox
 *          gtk_fixed (preview_image_canvas)	photoshare->preview_image_canvas
 *            gtk_event_box (preview_image_ebox)photoshare->preview_image_ebox
 *              gtk_image (preview_image)	photoshare->preview_image
 *    hbox (bottom_hbox)
 *      vbox (left_vbox)
 *        scrolled_window (hscrollbox)
 *          hbox (thumbnail_hbox)		photoshare->hscrollbox
 *        hbox (button_hbox)
 *          button (button_add)
 *          frame (button_tools)
 *            menu_bar (menubar_tools)
 *              menu_item (menu_head)
 *                menu (tools_menu) [create_preview_popup_menu]
 *          button (button_save)
 *          button (botton_close)
 *          label (counterlabel)		photoshare->counterlabel
 *
 *          toggle_button (button_friendsview)	photoshare->button_friendsview
 *      vbox (right_vbox)
 *        image (friend_view_image)		photoshare->friend_view_image
 */
void yphoto_gui(char *who)
{
	GtkWidget *photoshare_vbox;
	GtkWidget *scrollbox;
	GtkAdjustment *adj;
	GtkWidget *hscrollbox;
	GtkWidget *thumbnail_hbox;
	GtkWidget *preview_image_frame;
	GtkWidget *preview_image_canvas;
	GtkWidget *preview_image_ebox;
	GtkWidget *preview_image;
	GtkWidget *eboxprev;
	GtkWidget *bottom_hbox;
	GtkWidget *button_hbox;
	GtkWidget *left_vbox, *right_vbox;
	GtkWidget *friend_view_image;
	GtkWidget *button_add, *button_save, *button_close;
#if 0
	GtkToolItem *button_tools;
	GtkWidget *tools_icon;
#else
	GtkWidget *button_tools;
	GtkWidget *menubar_tools;
#endif
	GtkWidget *menu_head, *tools_menu;
	GtkWidget *pointer_item;
	GtkWidget *button_friendsview;
	GtkWidget *counterlabel;
	GtkWidget *pm_hbox_main;
	PM_SESSION *pm_sess = NULL;
	PHOTOSHARE *photoshare = NULL;

	if ( !(pm_sess = find_pm_session(who)) ) {
		/* open a new window if one doesn't exist already */
	        pm_sess=new_pm_session(who);
	}

	pm_hbox_main=g_object_get_data(G_OBJECT(pm_sess->pm_window), "hbox_main");
	if (!pm_hbox_main) {
		return;
	}

	// prepare list for pictures
	if (pm_sess->photoshare) {
		gtk_widget_destroy(pm_sess->photoshare->vbox);
	}

	pm_sess->photoshare = malloc(sizeof(PHOTOSHARE));
	memset(pm_sess->photoshare, 0, sizeof(PHOTOSHARE));
	photoshare = pm_sess->photoshare;
	/* paranoia check */
	if (!photoshare_dir) {
		photoshare_dir = strdup(getenv("HOME"));
		write_config();
	}

	// create and show the widgets
	photoshare_vbox = gtk_vbox_new(FALSE, 0);
	photoshare->vbox = photoshare_vbox;
	gtk_paned_pack2(GTK_PANED(pm_hbox_main), photoshare_vbox, TRUE, TRUE);

	/* Create the preview box. */
	eboxprev = gtk_event_box_new();	
	gtk_box_pack_start(GTK_BOX(photoshare_vbox), eboxprev, TRUE, TRUE, 0);

	preview_image_frame = gtk_vbox_new(FALSE, 0);
	photoshare->preview_image_frame = preview_image_frame;
	gtk_container_add(GTK_CONTAINER(eboxprev), preview_image_frame);

	scrollbox = gtk_scrolled_window_new(NULL, NULL);
	photoshare->scrollbox = scrollbox;
	gtk_container_add(GTK_CONTAINER(preview_image_frame), scrollbox);
	gtk_container_set_border_width(GTK_CONTAINER(scrollbox), 0);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);

	preview_image_canvas = gtk_fixed_new();
	photoshare->preview_image_canvas = preview_image_canvas;
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollbox), preview_image_canvas);

	preview_image_ebox = gtk_event_box_new();
	photoshare->preview_image_ebox = preview_image_ebox;
	gtk_fixed_put(GTK_FIXED(preview_image_canvas), preview_image_ebox, 0, 0);

	gtk_widget_add_events(preview_image_ebox,
			      GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);

	preview_image = gtk_image_new();
	gtk_container_add(GTK_CONTAINER(preview_image_ebox), preview_image);
	photoshare->preview_image  = preview_image;

	bottom_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(photoshare_vbox), bottom_hbox, FALSE, FALSE, 0);
	left_vbox  = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(bottom_hbox), left_vbox, TRUE, TRUE,  0);

	/* Create the thumbnail scrollbox. */
	hscrollbox = gtk_scrolled_window_new (NULL, NULL);
	gtk_box_pack_start(GTK_BOX(left_vbox), hscrollbox, FALSE, FALSE, 0);

	thumbnail_hbox = gtk_hbox_new(FALSE, 0);
	photoshare->hscrollbox = thumbnail_hbox;
	gtk_widget_set_size_request(thumbnail_hbox, -1, 55);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(hscrollbox), thumbnail_hbox);
	gtk_container_set_border_width(GTK_CONTAINER(hscrollbox),3);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(hscrollbox),
				       GTK_POLICY_ALWAYS,
				       GTK_POLICY_NEVER);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(hscrollbox), GTK_SHADOW_IN);
	
	gtk_drag_dest_set(hscrollbox, GTK_DEST_DEFAULT_ALL, NULL, 0, GDK_ACTION_COPY);
	gtk_drag_dest_add_uri_targets(hscrollbox);

	/* Create the botton box for the buttons.. */
	button_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(left_vbox), button_hbox, FALSE, FALSE, 0);

	button_add  = gtk_button_new_from_stock(GTK_STOCK_ADD);

	menu_head  = gtk_menu_item_new_with_label("Tools");
	tools_menu = create_preview_popup_menu(pm_sess);
	pointer_item = g_object_get_data(G_OBJECT(tools_menu), "pointer_item");
	if (pointer_item) {
		gtk_container_remove(GTK_CONTAINER(tools_menu), pointer_item);
	}

#if 0
	button_tools = gtk_option_menu_new();
	gtk_button_set_label(GTK_BUTTON(button_tools), "TOOLS");
	gtk_option_menu_set_menu(GTK_OPTION_MENU(button_tools), tools_menu);
#endif
#if 0
	tools_icon   = gtk_image_new_from_stock(GTK_STOCK_EXECUTE, GTK_ICON_SIZE_MENU);
	button_tools = gtk_menu_tool_button_new(NULL, "Tools");
	gtk_tool_item_set_is_important(button_tools, TRUE);
	gtk_tool_item_set_visible_horizontal(button_tools, TRUE);
	gtk_tool_item_set_visible_vertical(button_tools, TRUE);
	gtk_menu_tool_button_set_menu(GTK_MENU_TOOL_BUTTON(button_tools), tools_menu);
#endif
#if 1
	button_tools  = gtk_frame_new(NULL);
	menubar_tools = gtk_menu_bar_new();
	gtk_container_add(GTK_CONTAINER(button_tools), menubar_tools);
	gtk_container_add(GTK_CONTAINER(menubar_tools), menu_head);
	gtk_container_set_border_width(GTK_CONTAINER(menubar_tools), 0);
	gtk_container_set_border_width(GTK_CONTAINER(menu_head), 0);
//	gtk_container_add (GTK_CONTAINER(button_tools), menubar_tools);
	gtk_menu_item_set_submenu (GTK_MENU_ITEM(menu_head), tools_menu);
#endif

	button_save  = gtk_button_new_from_stock(GTK_STOCK_SAVE);
	button_close = gtk_button_new_from_stock(GTK_STOCK_CLOSE);

	counterlabel = gtk_label_new(NULL);
	photoshare->counterlabel = counterlabel;

	button_friendsview = gtk_toggle_button_new_with_label("Friend's View >");
	photoshare->button_friendsview = button_friendsview;
	gtk_toggle_button_set_active((GtkToggleButton *)button_friendsview, FALSE);

	gtk_box_pack_start(GTK_BOX(button_hbox), button_add,   FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(button_hbox), GTK_WIDGET(button_tools), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(button_hbox), button_save,  FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(button_hbox), button_close, FALSE, FALSE, 2);
	gtk_box_pack_start(GTK_BOX(button_hbox), counterlabel, FALSE, FALSE, 0);
	gtk_box_pack_end(GTK_BOX(button_hbox), button_friendsview, FALSE, FALSE, 0);

	/* now pack the above boxen into a vbox, and pack that vbox into a hbox. */
	right_vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(bottom_hbox), right_vbox, FALSE, FALSE, 0);

	friend_view_image = gtk_image_new();
	photoshare->friend_view_image = friend_view_image;
	gtk_widget_set_size_request(friend_view_image, 55, 55);
	gtk_box_pack_start(GTK_BOX(right_vbox), friend_view_image, FALSE, FALSE, 0);

	gtk_widget_show_all(photoshare_vbox);

	// connect signals to callback functions
	gtk_drag_dest_set(eboxprev, GTK_DEST_DEFAULT_ALL, NULL, 0, GDK_ACTION_COPY);
	gtk_drag_dest_add_uri_targets(eboxprev);

	g_signal_connect(photoshare_vbox, "destroy",           G_CALLBACK(on_mainwindow_destroy),  pm_sess);
	g_signal_connect(eboxprev,     "drag-data-received",   G_CALLBACK(on_drag_data_received),  pm_sess);
	g_signal_connect(eboxprev,     "button-press-event",   G_CALLBACK(on_preview_clicked),     pm_sess);
	g_signal_connect(hscrollbox,   "drag-data-received",   G_CALLBACK(on_drag_data_received),  pm_sess);
	g_signal_connect(button_add,   "clicked",       G_CALLBACK(on_button_add_clicked),         pm_sess);
//	g_signal_connect(button_tools, "clicked",       G_CALLBACK(on_button_tools_clicked),       pm_sess);
	g_signal_connect(button_close, "clicked",       G_CALLBACK(on_button_close_clicked),       pm_sess);
	g_signal_connect(button_save,  "clicked",       G_CALLBACK(on_preview_save_activate),      pm_sess);
	g_signal_connect(button_friendsview, "clicked", G_CALLBACK(on_button_friendsview_clicked), pm_sess);

	g_signal_connect(preview_image_ebox, "enter-notify-event",  G_CALLBACK(on_image_enter),    pm_sess);
	g_signal_connect(preview_image_ebox, "leave-notify-event",  G_CALLBACK(on_image_leave),    pm_sess);

	adj = gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW(scrollbox));
	g_signal_connect(adj,       "value-changed", G_CALLBACK(yphoto_hadjustment_value_changed), pm_sess);
	g_signal_connect(adj,       "changed",       G_CALLBACK(yphoto_hadjustment_changed),       pm_sess);
	adj = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW(scrollbox));
	g_signal_connect(adj,       "value-changed", G_CALLBACK(yphoto_vadjustment_value_changed), pm_sess);
	g_signal_connect(adj,       "changed",       G_CALLBACK(yphoto_vadjustment_changed),       pm_sess);

	if (!yphoto_cursor_tcross) {
		yphoto_cursor_tcross = gdk_cursor_new(GDK_TCROSS);
	}

} /* yphoto_gui */

void on_yphoto_reject(GtkWidget *button, gpointer user_data) {
	PM_SESSION *pm_sess = (PM_SESSION *)user_data;
	GtkWidget *tmp_widget;
	
	ymsg_yphoto_reject(ymsg_sess, pm_sess->pm_user);

	tmp_widget = g_object_get_data(G_OBJECT(button), "mywindow");
	if (tmp_widget) gtk_widget_destroy(tmp_widget);
}


/*
 *    Packet interface methods
 */

void on_yphoto_accept(GtkWidget *button, gpointer user_data)
{
	PM_SESSION *pm_sess = (PM_SESSION *)user_data;
	GtkWidget *tmp_widget;
	
	ymsg_yphoto_accept(ymsg_sess, pm_sess->pm_user);

	pm_notebook_select_tab(pm_sess->pm_notebook, pm_sess->pm_user);
	yphoto_gui(pm_sess->pm_user);

	tmp_widget = g_object_get_data(G_OBJECT(button), "mywindow");
	if (tmp_widget) gtk_widget_destroy(tmp_widget);
}

/* handles incoming photosharing-session */
void yahoo_yphoto_offer_msg(char *who)
{
	char buff[512];
	GtkWidget *okbutton=NULL;
	GtkWidget *cbutton=NULL;
	int accept_pm;
	PM_SESSION *pm_sess;
	GtkWidget *parent;

	accept_pm = get_pm_perms(who);
	if ( ! accept_pm) {					
		send_rejection_message(0);
		return;
	}

	if ( !(pm_sess = find_pm_session(who)) ) {
		/* open a new window if one doesn't exist already */
	        pm_sess=new_pm_session(who);
	}
	parent = pm_sess->pm_notebook->window;
	if ( auto_raise_pm ) {
		focus_pm_entry(pm_sess);
	}

	//TODO: looking for options (pm's off: photosharing off)
	
	snprintf(buff, sizeof(buff)-1, _("The Yahoo user <b>%s</b> wants to share photos.\n\nDo you want to accept?\n"), who);

	okbutton = show_confirm_dialog_config_p(parent, buff, _("Yes"), _("No"), 0);
	if (!okbutton) {
		ymsg_yphoto_reject(ymsg_sess, who); 
		return;
	}

	g_signal_connect(G_OBJECT(okbutton),  "clicked", G_CALLBACK(on_yphoto_accept),  pm_sess);

	cbutton = g_object_get_data(G_OBJECT(okbutton), "cancel");
	if (cbutton) {
		g_signal_connect(G_OBJECT(cbutton),  "clicked", G_CALLBACK(on_yphoto_reject), pm_sess);
	}
	play_sound_event(SOUND_EVENT_OTHER);
} /* yahoo_yphoto_offer */


/* sends invitation for a photosharing-session */
void yahoo_yphoto_invite(char *who)
{
	char *buf;
	char *invite_strings[] = {
		"  ",
		YAHOO_STYLE_BOLDON,
		YAHOO_COLOR_BLUE,
		"**",
		YAHOO_COLOR_RED,
		" ",
		_("Buddy"),
		": ",
		YAHOO_COLOR_PURPLE,
		who,
		YAHOO_COLOR_BLUE,
		"  ",
		_("has been invited to share photos"),
		". **",
		YAHOO_COLOR_BLACK,
		YAHOO_STYLE_BOLDOFF,
		"\n",
		0
	};

	ymsg_yphoto_invite(ymsg_sess, who);

	if (chat_timestamp && enter_leave_timestamp) {
		append_timestamp(chat_window, NULL);
		append_to_textbox(chat_window, NULL, " ");
	}

	append_char_icon_text(NULL, GYACHI_PHOTOSHARE, GTK_ICON_SIZE_BUTTON);

	buf = build_string(invite_strings);
	append_to_textbox(chat_window, NULL, buf);
	append_to_open_pms(who, buf, GYACHI_PHOTOSHARE, 1);
	free(buf);
	buf=NULL;
	play_sound_event(SOUND_EVENT_OTHER);
} /* yahoo_yphoto_invite */


/* handles declining of session */
void yahoo_yphoto_decline_msg(char *who)
{
	GtkWidget *parent;
	char status_msg[512];

	parent = find_pms_window(who);
	snprintf(status_msg, 500, _("%s has declined photosharing-session"), who);
	show_ok_dialog_p(parent, status_msg);
}


/* buddy accepted invitation */
void yahoo_yphoto_accept_msg(char *who)
{
	yphoto_gui(who);	
}


/* handles closing of session */
void yahoo_yphoto_closed_msg(char *who)
{
	PM_SESSION *pm_sess;
	PHOTOSHARE *photoshare;
	GtkWidget *parent;
	GtkWidget  *pm_session;
	GtkWidget  *dlmanager;
	char status_msg[512];

	pm_sess = find_pm_session(who);
	if (pm_sess == 0) {
		return;
	}

	photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	if (photoshare) {
		parent = pm_sess->pm_notebook->window;

		pm_notebook_select_tab(pm_sess->pm_notebook, pm_sess->pm_user);
		snprintf(status_msg, 500, _("%s has closed photosharing-session"), who);
		show_ok_dialog_p(parent, status_msg);

		gtk_widget_destroy(photoshare->vbox);
	}

	/* close the d/l manager here */
	pm_session = pm_sess->pm_window;
	dlmanager = g_object_get_data(G_OBJECT(pm_session), "dlmanager");
	if (dlmanager) {
		gtk_widget_hide(dlmanager);
	}
}

/* handle preview packet 0xd7/215 */
void yahoo_yphoto_preview()
{
	PM_SESSION *pm_sess;
	GtkWindow *parent;
	PHOTOSHARE *photoshare;
	int  num;
	char *key   = 0;
	char *value = 0;
	char *who             = 0;
	char *filekey         = 0;
	char *hint            = 0;
	char *filename        = 0;
	char *encoded_preview = 0;
	char *decoded_preview = 0;
	char *ptr;
	int  encoded_length;
	int  decoded_length;
	GList *current_thumb;

	for (num=0; ymsg_field_pair(num, &key, &value); num++) {
		if (!strcmp(key, "4") || !strcmp(key, "0")) { /* buddy id */
			who = value;
			continue;
		}

		if (!strcmp(key, "5")) { /* our id */
			continue;
		}
		if (!strcmp(key, "140")) { /* always seems to be "1" */
			continue;
		}
		if (!strcmp(key, "245")) { /* uniq file key */
			filekey = value;
			continue;
		}
		if (!strcmp(key, "246")) { /* file name */
			filename = value;
			continue;
		}
		if (!strcmp(key, "267")) { /* preview image */
			encoded_preview = value;
			continue;
		}

		if (!strcmp(key, "10093")) { /* dunno, always has "4" for value */
			continue;
		}

		/* unknown attribute. We should log these, and then figure out
		 * what they mean...
		 */
		if ( capture_fp) {	
			fprintf(capture_fp, "*** DEBUG: PHOTOSHARE PREV (0xd7/215) TYPE: KEY: %s, VALUE:%s\n",
				key,
				value);
		}
	}

	if (!filekey || !who) {
		/* if no filekey, or no buddy id, we can't process */
		return;
	}

	/*
	 * 1) verify that we're in photosharing mode.
	 *    - ignore if we're not (i.e. no PHOTOSHARE struct setup).
	 * 2) populate photo_info from filekey/prev values (and decode the preview).
	 *    - if no preview, then don't add to the preview list!
	 * 3) ack the packet with:
	 *	    4	buddy id
	 *	    5	our id
	 *	  140	1
	 *	  245	file handle
	 *	10093	4
	 */

	pm_sess = find_pm_session(who);
	if (pm_sess == 0) {
		return;
	}
	parent = (GtkWindow *)pm_sess->pm_notebook->window;

	photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	if (photoshare == 0) {
		return;
	}

	if (encoded_preview) {
		/* - decode preview
		 * - load into a pixbuf
		 * - add pixbuf to the picture list.
		 * - *might* need to save as an intermediate file, until the real file arrives
		 */

		encoded_length = strlen(encoded_preview);
		decoded_length = encoded_length;
		decoded_preview = malloc(decoded_length); /* strickly speaking, should be 3/4 of this */
		decoded_length = yphoto_b64_decode(decoded_preview, decoded_length, encoded_preview, encoded_length);
		if (decoded_length < 0) {
			/* error occured. No idea what to do... Figure out later...*/
		}

		hint = filename;
		filename = write_to_file(decoded_preview, decoded_length, filekey, (GtkWidget *)parent);
		current_thumb = photoshare->current_thumb;
		yphoto_add_file(pm_sess, filename, hint, filekey);

		if (photoshare->current_thumb) {
			/* update the counters */
			set_counterlabel(pm_sess);
		}

		/* hmm... probably should to a pm_notebook typing status, so
		 * that if tab not selected, we start blinking it...
		 * FIXME
		 */
		free(decoded_preview);
		free(filename);

		/* send the 215 ACK */
		ymsg_yphoto_prev_ack(ymsg_sess, pm_sess->pm_user, filekey);

		/* send the 218 xchange info request
		 * Strip the training extension, if it is there...
		 */
		ptr=filekey+24;
		if (strlen(filekey) > 24) {
			if (*ptr == '.') {
				*ptr = 0;
			}
		}

		ymsg_yphoto_trans_init(ymsg_sess, pm_sess->pm_user, filekey);

		refresh_buddy_pic(pm_sess);
	}
}

/* handle yphoto_key packet 0xd8/216 */
void yahoo_yphoto_key()
{
	PM_SESSION *pm_sess;
	PHOTOSHARE *photoshare;
	char  *key   = 0;
	char  *value = 0;
	char  *who     = 0;
	char  *filekey = 0;
	char  *preview = 0;
	int    num;

	for (num=0; ymsg_field_pair(num, &key, &value); num++) {
		if (!strcmp(key, "4") || !strcmp(key, "0")) { /* buddy id */
			who = value;
			continue;
		}

		if (!strcmp(key, "5")) { /* our id */
			continue;
		}

		if (!strcmp(key, "140")) { /* always seems to be "1" */
			continue;
		}

		if (!strcmp(key, "245")) { /* uniq file key */
			filekey = value;
			continue;
		}

		if (!strcmp(key, "194")) { /* image seq number */
			preview = value;
			continue;
		}

		if (!strcmp(key, "10093")) { /* dunno, always has "4" for value */
			continue;
		}

		/* unknown attribute. We should log these, and then figure out
		 * what they mean...
		 */
		if ( capture_fp) {	
			fprintf(capture_fp, "*** DEBUG: PHOTOSHARE PREV (0xd7/215) TYPE: KEY: %s, VALUE:%s\n",
				key,
				value);
		}
	}

	if (!filekey || !who) {
		/* if no filekey, or no buddy id, we can't process */
		return;
	}

	/*
	 * 1) verify that we're in photosharing mode.
	 *    - ignore if we're not (i.e. no PHOTOSHARE struct setup).
	 */

	pm_sess = find_pm_session(who);
	if (pm_sess == 0) {
		return;
	}

	photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	if (photoshare == 0) {
		return;
	}

	/* our buddy is telling us which picture he is looking at.
	 * Both the index number, as well as the filekey is sent.
	 * Store the filekey as buddy_filekey, and display the buddy pic, if it
	 * has been sent.
	 */

	if (photoshare->buddy_filekey) {
		free(photoshare->buddy_filekey);
	}
	photoshare->buddy_filekey = strdup(filekey);

	refresh_buddy_pic(pm_sess);
}

/* handle yphoto_remove 0xd9/217 */
void yahoo_yphoto_remove()
{
	PM_SESSION *pm_sess;
	PHOTOSHARE *photoshare;
	PHOTO_INFO *photo_info;
	char  *key   = 0;
	char  *value = 0;
	char  *who     = 0;
	char  *filekey = 0;
	int    num;

	for (num=0; ymsg_field_pair(num, &key, &value); num++) {
		if (!strcmp(key, "4") || !strcmp(key, "0")) { /* buddy id */
			who = value;
			continue;
		}

		if (!strcmp(key, "5")) { /* our id */
			continue;
		}

		if (!strcmp(key, "140")) { /* always seems to be "1" */
			continue;
		}

		if (!strcmp(key, "245")) { /* uniq file key */
			filekey = value;
			continue;
		}

		if (!strcmp(key, "10093")) { /* dunno, always has "4" for value */
			continue;
		}

		/* unknown attribute. We should log these, and then figure out
		 * what they mean...
		 */
		if ( capture_fp) {	
			fprintf(capture_fp, "*** DEBUG: PHOTOSHARE PREV (0xd7/215) TYPE: KEY: %s, VALUE:%s\n",
				key,
				value);
		}
	}

	if (!filekey || !who) {
		/* if no filekey, or no buddy id, we can't process */
		return;
	}

	/*
	 * 1) verify that we're in photosharing mode.
	 *    - ignore if we're not (i.e. no PHOTOSHARE struct setup).
	 */

	pm_sess = find_pm_session(who);
	if (pm_sess == 0) {
		return;
	}

	photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	if (photoshare == 0) {
		return;
	}

	/* our buddy is telling us which picture he is removing. We
	 * need to remove this photo, and then send a new photo_key on the
	 * next selected photo. This is already done as part of the
	 * photo_remove code.
	 */

	photo_info = find_info_by_filekey(pm_sess, filekey);
	if (!photo_info) {
		return;
	}

	remove_photo(pm_sess, photo_info->my_element);
}


void yphoto_send_trans_ack(void *arg)
{
	FILE_SEND_INFO *file_send_info;
	PHOTO_INFO *photo_info;
	SENDFILE_QUEUE_INFO *sendfile_queue_info;
	GList *element;

	file_send_info = arg;
	photo_info = file_send_info->photo_info;

	/* start the file upload */
	ymsg_yphoto_trans_ack(ymsg_sess,
			  file_send_info->who,
			  photo_info->filekey,
			  photo_info->file_token,
			  photo_info->filesize,
			  file_send_info->host_ip);
	free(file_send_info->host_ip);
	free(file_send_info->who);
	free(file_send_info);

	for (element=sendfile_queue; element; element=element->next) {
		sendfile_queue_info = (SENDFILE_QUEUE_INFO *)element->data;
		if (sendfile_queue_info->photo_info == photo_info) {
			free(sendfile_queue_info);
			sendfile_queue = g_list_delete_link(sendfile_queue, element);
			break;
		}
	}

	start_next_sendfile();
}

void yphoto_rename_photo(void *arg)
{
	FILE_RECEIVE_INFO *file_receive_info=arg;
	PHOTO_INFO *photo_info = file_receive_info->photo_info;
	PM_SESSION *pm_sess    = file_receive_info->pm_sess;
	PHOTOSHARE *photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	char       *old_filename;
	GdkPixbufFormat *fileinfo;
	int         height;
	int         width;
	char       *extn, *ptr;
	char      **extns;
	GtkImage   *thumbimage;

	if (is_valid_photo_info(pm_sess, photo_info)) {
		old_filename = malloc(strlen(photo_info->filename)+2);
		strcpy(old_filename, photo_info->filename);
		strcat(old_filename, "~");

		fileinfo = gdk_pixbuf_get_file_info(old_filename, &width, &height);
		if (fileinfo) {
			photo_info->height = height;
			photo_info->width  = width;

			/* fix filename extension !!! */
			extns = gdk_pixbuf_format_get_extensions(fileinfo);
			if (extns) {
				if (!strcmp(*extns, "jpe") || !strcmp(*extns, "jpeg")) {
					extn = strdup("jpg");
				}
				else {
					extn = strdup(*extns);
				}
				g_strfreev(extns);

				ptr = malloc(strlen(photo_info->filename) + strlen(extn) + 2);
				strcpy(ptr, photo_info->filename);
				strcat(ptr, ".");
				strcat(ptr, extn);
				unlink(photo_info->filename);
				free(photo_info->filename);
				photo_info->filename = ptr;

				ptr = malloc(strlen(photo_info->hint) + strlen(extn) + 2);
				strcpy(ptr, photo_info->hint);
				strcat(ptr, ".");
				strcat(ptr, extn);
				free(photo_info->hint);
				photo_info->hint = ptr;

				free(extn);

				thumbimage = g_object_get_data(G_OBJECT(photo_info->button), "thumbimage");
				if (thumbimage) {
					gyachi_set_tooltip((GtkWidget*)thumbimage, photo_info->hint);
				}
			}
		}

		rename(old_filename, photo_info->filename);
		free(old_filename);
	}
	else {
		/* FIXME -- probably need to unlink the original, or the new, or both filename */
	}

	free(arg);

	if (photoshare && (photoshare->current_thumb == photo_info->my_element)) {
		redraw_selected_icon_image(pm_sess, 0);
	}
}


/* handle yphoto_trans packet 0xda/218
 *
 * There are 2 circumstances when we receive a 218 packet.
 * The first is when our buddy requests a file transfer packet.
 *     We will response back with a 218 ack packet that includes the file token, filesize, and relay host.
 * The 2nd case we will see a 218 packet is when we initiated the transfer, and we see the return
 *     218 ack packet.
 * We can distinguish between the 2 packets by the presence of fields 250 (relay_host), 253 (redundant
 * file_handle), and 276 (another file token, which doesn't seem to be used any place).
 * Any of these 3 indicate that this is the 218 ack response. Otherwise it is the 218 inquire packet.
 * We use the presence of the 250 field (relay host) to indicate whether this is a request, or an ACK.
 * The 250 field will only be present in the ACK.
 */
void yahoo_yphoto_trans()
{
	PM_SESSION *pm_sess;
	PHOTOSHARE *photoshare;
	PHOTO_INFO *photo_info;
	FILE_INFO  *file_info;
	char  *key    = 0;
	char  *value  = 0;
	char  *who        = 0;
	char  *filekey    = 0;
	char  *file_token = 0;
	char  *filesize   = 0;
	char  *relay_host = 0;
	int    num;

	for (num=0; ymsg_field_pair(num, &key, &value); num++) {
		if (!strcmp(key, "4") || !strcmp(key, "0")) { /* buddy id */
			who = value;
			continue;
		}

		if (!strcmp(key, "5")) { /* our id */
			continue;
		}

		if (!strcmp(key, "28")) { /* filesize */
			filesize = value;
			continue;
		}

		if (!strcmp(key, "32")) { /* always seems to be "1" for inquiry, "18" for reply */
			continue;
		}

		if (!strcmp(key, "140")) { /* always seems to be "1" */
			continue;
		}

		if (!strcmp(key, "245")) { /* uniq file key */
			filekey = value;

#if 0
{
  int   decoded_length;
  char *decoded_buffer, *p;
  char *encoded_buffer = strdup(filekey);
  decoded_length = strlen(filekey);
  decoded_buffer = malloc(decoded_length);
  mac64_to_b64(encoded_buffer);
  decoded_length = yphoto_b64_decode(decoded_buffer, decoded_length, encoded_buffer, decoded_length);
  decoded_buffer[decoded_length]=0;
  printf("filekey: %s\n", filekey);
  printf("decoded: "); for (p=decoded_buffer; *p ; p++) printf("%02x ", *p); printf("\n");
 }
#endif
			continue;
		}

		if (!strcmp(key, "248")) { /* always seems to be "2" */
			continue;
		}

		if (!strcmp(key, "249")) { /* always seems to be "3" */
			continue;
		}

		if (!strcmp(key, "250")) { /* relay_host. Used to upload/download the file */
			relay_host = value;
			continue;
		}

		if (!strcmp(key, "251")) { /* file_token assigned by yahoo. Used to upload/download the file */
			file_token = value;
			continue;
		}

		if (!strcmp(key, "253")) { /* redundant copy of the filekey */
			continue;
		}

		if (!strcmp(key, "276")) { /* another file token. doesn't seem to be used */
			continue;
		}

		if (!strcmp(key, "10093")) { /* dunno, always has "4" for value */
			continue;
		}

		/* unknown attribute. We should log these, and then figure out
		 * what they mean...
		 */
		if ( capture_fp) {	
			fprintf(capture_fp, "*** DEBUG: PHOTOSHARE PREV (0xd7/215) TYPE: KEY: %s, VALUE:%s\n",
				key,
				value);
		}
	}

	if (!filekey || !who || !file_token) {
		/* if no filekey, file_token, or no buddy id, we can't process */
		return;
	}

	/*
	 * 1) verify that we're in photosharing mode.
	 *    - ignore if we're not (i.e. no PHOTOSHARE struct setup).
	 */

	pm_sess = find_pm_session(who);
	if (pm_sess == 0) {
		return;
	}

	photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	if (photoshare == 0) {
		return;
	}

	/*
	 * 2) verify that we have a filekey, and that we have a corresponding
	 *    photo_info entry with that filekey. Otherwise ignore.
	 *    NOTE: the 218 request always seems to be sent AFTER the PREVIEW and KEY
	 *    packets. This is important, as the PREVIEW packet is the one that will
	 *    create the photo_info for this photo.
	 */

	photo_info = find_info_by_filekey(pm_sess, filekey);
	if (!photo_info) {
		return;
	}

	/*
	 * 3) see who owns the file. If the buddy owns the file, we should't really be getting
	 *    a 218 request. Likewise, if we own the file, we shouldn't be getting a 218 ACK.
	 */

	switch (photo_info->direction) {
	case RECEIVING:
		/* we're receiving the file. This SHOULD be in the 218 reply/ack packet */
		if (!relay_host) {
			/* This is a reply/ack packet (i.e. relay_host field SHOULD exist, but doesn't). */
			return;
		}
		break;

	case SENDING:
		/* we're the file sender. This SHOULD NOT be the 218 ACK packet */
		if (relay_host) {
			/* This is a request packet (i.e. relay_host field SHOULD NOT exist, but does!). */
			return;
		}
		break;

	default:
		return;
	}

	/* Sanity checking done. Now to handle the 2 different cases. */
	if (!relay_host) {
		FILE_SEND_INFO *file_send_info;

		/* Handle 218 request packet. Generate the ack, and start upload. */
		if (photo_info->file_token) {
			/* some sanity checking. In reality, we should never see TWO request packets, so
			 * we would never ALREADY have a file_token stored. But... Never hurts to check/
			 */
			free(photo_info->file_token);
		}
		photo_info->file_token = strdup(file_token);

		file_send_info = malloc(sizeof(FILE_SEND_INFO));
		file_send_info->host_ip    = NULL;
		file_send_info->who        = strdup(who);
		file_send_info->photo_info = photo_info;

		file_info=create_fxfile_info(who, photo_info->hint, photo_info->filekey, photo_info->filesize);
		file_info->hostname      = strdup(YAHOO_RELAYSERVER);
		file_info->token         = strdup(photo_info->file_token);
		file_info->fullfilename  = strdup(photo_info->filename);
		file_info->open_callback = yphoto_send_trans_ack;
		file_info->open_arg      = file_send_info;

		g_thread_create(yahoo_fxfer_sendfile_thread, file_info, FALSE, NULL);
	}
	else {
		FILE_RECEIVE_INFO *file_receive_info;

		/* Handle 218 ACK packet. Start download. */
		if (photo_info->file_token) {
			/* some sanity checking. In reality, we should never see TWO request packets, so
			 * we would never ALREADY have a file_token stored. But... Never hurts to check/
			 */
			free(photo_info->file_token);
		}
		photo_info->file_token = strdup(file_token);
		photo_info->filesize = atoi(filesize);

		/* start file d/l */
		file_receive_info = malloc(sizeof(FILE_RECEIVE_INFO));
		file_receive_info->pm_sess = pm_sess;
		file_receive_info->photo_info = photo_info;
		file_info=create_fxfile_info(who, photo_info->hint, photo_info->filekey, photo_info->filesize);
		file_info->fullfilename=malloc(strlen(photo_info->filename)+2);
		strcpy(file_info->fullfilename, photo_info->filename);
		strcat(file_info->fullfilename, "~");
		file_info->server          = strdup(relay_host);
		file_info->token           = strdup(photo_info->file_token);
		file_info->finish_callback = yphoto_rename_photo;
		file_info->finish_arg      = file_receive_info;

		g_thread_create(yahoo_fxfer_getfile_thread, file_info, FALSE, NULL);
	}
}


/* handle yphoto_key packet 0xdb/219

   packet type:    219 / 0xdb / YMSG_YPHOTO_POINTER

   Turn pointer on at coordinate

    Sender sends:

                1       our id
                5       sender id
                245     file handle. e.g. BFuArmUZ5YSh8g8jlrZHhw--
                272     X coordinate (upper left is 0,0)
                273     Y coordinate
                13      1
                194     ## (a sequential counter)

    Receiver will see:

                0       buddy id
                1       buddy id
                4       buddy id
                5       our id
                13      1
                32      1 (status)
                140     1
                194     ## (a sequential counter, to be ignored)
                245     file handle. e.g. BFuArmUZ5YSh8g8jlrZHhw--
                272     X coordinant (upper left is 0,0)
                273     Y coordinant
                10093   4


   Turn pointer off at coordinate

    Sender sends:

                1       our id
                5       sender id
                245     file handle. e.g. BFuArmUZ5YSh8g8jlrZHhw--
                272     X coordinate (upper left is 0,0)
                273     Y coordinate
                13      0
                194     ## (a sequential counter)

    Receiver will see:

                0       buddy id
                1       buddy id
                4       buddy id
                5       our id
                13      0
                32      1 (status)
                140     1
                194     ## (a sequential counter, to be ignored)
                245     file handle. e.g. BFuArmUZ5YSh8g8jlrZHhw--
                272     X coordinant (upper left is 0,0)
                273     Y coordinant
                10093   4
*/

void yahoo_yphoto_pointer()
{
	PM_SESSION *pm_sess;
	PHOTOSHARE *photoshare;
	PHOTO_INFO *photo_info;
	char  *key    = 0;
	char  *value  = 0;
	char  *who        = 0;
	char  *filekey    = 0;
	int    x;
	int    y;
	int    on_off;
	int    num;

	for (num=0; ymsg_field_pair(num, &key, &value); num++) {
		if (!strcmp(key, "4") || !strcmp(key, "0")) { /* buddy id */
			who = value;
			continue;
		}

		if (!strcmp(key, "5")) { /* our id */
			continue;
		}

		if (!strcmp(key, "13")) { /* 1 == pointer is on, 0 == pointer is off */
			on_off = atoi(value);
			continue;
		}

		if (!strcmp(key, "32")) { /* seems to be status. seems to be always "1" */
			continue;
		}

		if (!strcmp(key, "140")) { /* always seems to be "1" */
			continue;
		}

		if (!strcmp(key, "194")) { /* This is a counter. Ignore. */
			continue;
		}

		if (!strcmp(key, "245")) { /* uniq file key */
			filekey = value;

#if 0
{
  int   decoded_length;
  char *decoded_buffer, *p;
  char *encoded_buffer = strdup(filekey);
  decoded_length = strlen(filekey);
  decoded_buffer = malloc(decoded_length);
  mac64_to_b64(encoded_buffer);
  decoded_length = yphoto_b64_decode(decoded_buffer, decoded_length, encoded_buffer, decoded_length);
  decoded_buffer[decoded_length]=0;
  printf("filekey: %s\n", filekey);
  printf("decoded: "); for (p=decoded_buffer; *p ; p++) printf("%02x ", *p); printf("\n");
 }
#endif
			continue;
		}

		if (!strcmp(key, "272")) { /* X coordinate */
			x = atoi(value);
			continue;
		}

		if (!strcmp(key, "273")) { /* Y coordinate */
			y = atoi(value);
			continue;
		}

		if (!strcmp(key, "10093")) { /* dunno, always has "4" for value */
			continue;
		}

		/* unknown attribute. We should log these, and then figure out
		 * what they mean...
		 */
		if ( capture_fp) {	
			fprintf(capture_fp, "*** DEBUG: PHOTOSHARE POINTER (0xdb/219) TYPE: KEY: %s, VALUE:%s\n",
				key,
				value);
		}
	}

	if (!filekey || !who) {
		/* if no filekey, or no buddy id, we can't process */
		return;
	}

	/*
	 * 1) verify that we're in photosharing mode.
	 *    - ignore if we're not (i.e. no PHOTOSHARE struct setup).
	 */

	pm_sess = find_pm_session(who);
	if (pm_sess == 0) {
		return;
	}

	photoshare = (PHOTOSHARE *)pm_sess->photoshare;
	if (photoshare == 0) {
		return;
	}

	/*
	 * 2) verify that we have a filekey, and that we have a corresponding
	 *    photo_info entry with that filekey. Otherwise ignore.
	 *    NOTE: the 218 request always seems to be sent AFTER the PREVIEW and KEY
	 *    packets. This is important, as the PREVIEW packet is the one that will
	 *    create the photo_info for this photo.
	 */

	photo_info = find_info_by_filekey(pm_sess, filekey);
	if (!photo_info) {
		return;
	}

	photo_info->pointer_on = on_off;
	photo_info->pointer_x = x;
	photo_info->pointer_y = y;

	redraw_selected_icon_image(pm_sess, 0);
}

//TODO:
// handling of cancel-msg
// logging and information in mainchat-window

