/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.common.medias;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.videolan.vlma.common.programs.IVlProgram;

/**
 * A satellite channel.
 * 
 * @author SylV
 * @version 1.0
 */
public class VlSatChannel extends VlMedia {

    private static final Logger logger = Logger.getLogger(VlSatChannel.class);

    private int frequency;

    private char polarisation;

    private int sid;

    private String fullname;

    private String country;

    private String category;

    private String encryption;

    private int errorCorrection;

    private int symbolRate;

    private String coverage;

    public static final String SAT_SAP_GROUP = "Satellite";

    public static final String RADIO_SAP_GROUP = "Radio";

    /**
     * Gets the channel category (TV, radio, encrypted, etc.).
     * 
     * @return the channel category
     */
    public String getCategory() {
        return category;
    }

    /**
     * Sets the channel category.
     * 
     * @param category
     *            the channel category to set
     */
    public void setCategory(String category) {
        this.category = category;
        updateSapGroup();
    }

    /**
     * Sets the channel program and updates its SAP group.
     * 
     * @param program
     *            the program associated to the media
     */
    public void setProgram(IVlProgram program) {
        super.setProgram(program);
        updateSapGroup();
    }

    /**
     * Updates the SAP group.
     * 
     */
    protected void updateSapGroup() {
        if (program != null) {
            if (category.equals("R-DIG"))
                program.setSapGroup(RADIO_SAP_GROUP);
            else
                program.setSapGroup(SAT_SAP_GROUP);
        }
    }

    /**
     * Gets the channel country.
     * 
     * @return the country
     */
    public String getCountry() {
        return country;
    }

    /**
     * Sets the channel country.
     * 
     * @param country
     *            the country to set
     */
    public void setCountry(String country) {
        this.country = country;
    }

    /**
     * Gets the channel encryption.
     * 
     * @return the channel encryption
     */
    public String getEncryption() {
        return encryption;
    }

    /**
     * Sets the encryption of the channel.
     * 
     * @param encryption
     *            the encryption to set
     */
    public void setEncryption(String encryption) {
        this.encryption = encryption;
    }

    /**
     * Gets the error correction.
     * 
     * @return the error correction
     */
    public int getErrorCorrection() {
        return errorCorrection;
    }

    /**
     * Sets the error correction.
     * 
     * @param errorCorrection
     *            the error correction to set
     */
    public void setErrorCorrection(int errorCorrection) {
        this.errorCorrection = errorCorrection;
    }

    /**
     * Gets the channel frequency.
     * 
     * @return the channel frequency
     */
    public int getFrequency() {
        return frequency;
    }

    /**
     * Sets the channel frequency.
     * 
     * @param frequency
     *            the frequency to set
     */
    public void setFrequency(int frequency) {
        this.frequency = frequency;
    }

    /**
     * Gets the channel polarisation.
     * 
     * @return the channel polarisation
     */
    public char getPolarisation() {
        return polarisation;
    }

    /**
     * Sets the channel polarisation.
     * 
     * @param polarisation
     *            the channel polarisation
     */
    public void setPolarisation(char polarisation) {
        this.polarisation = polarisation;
    }

    /**
     * Gets the channel SID.
     * 
     * @return the SID
     */
    public int getSid() {
        return sid;
    }

    /**
     * Sets the channel SID.
     * 
     * @param sid
     *            the SID to set.
     */
    public void setSid(int sid) {
        this.sid = sid;
    }

    /**
     * Gets the symbol rate.
     * 
     * @return the rate
     */
    public int getSymbolRate() {
        return symbolRate;
    }

    /**
     * Sets the symbol rate.
     * 
     * @param symbolRate
     *            the symbol rate to set
     */
    public void setSymbolRate(int symbolRate) {
        this.symbolRate = symbolRate;
    }

    /**
     * Gets the channel fullname.
     * 
     * @return the channel fullname
     */
    public String getFullname() {
        return fullname;
    }

    /**
     * Sets the channel fullname.
     * 
     * @param fullname
     *            the name to set
     */
    public void setFullname(String fullname) {
        this.fullname = fullname;
    }

    /**
     * Gets the channel coverage zone.
     * 
     * @return the coverage zone
     */
    public String getCoverage() {
        return coverage;
    }

    /**
     * Sets the coverage zone.
     * 
     * @param coverage
     *            the coverage zone to set
     */
    public void setCoverage(String coverage) {
        this.coverage = coverage;
    }

    /**
     * Compares this object against the specified object. The result is true 
     * if and only if the argument has same coverage zone, same frequency and 
     * same name.
     * 
     * @return true if objects are the same; false otherise
     */
    public boolean equals(Object o) {
        if (o.getClass().equals(VlSatChannel.class)) {
            VlSatChannel c = (VlSatChannel) o;
            logger.log(Level.DEBUG, "Comparison of " + getName() + " with "
                    + c.getName());
            return (c.getCoverage().equals(this.coverage)
                    && c.getFrequency() == this.frequency && c.name
                    .equals(this.name));
        }
        return false;
    }

    public boolean sameGroup(IVlMedia media) {
        if (media.getClass() == VlSatChannel.class) {
            VlSatChannel c = (VlSatChannel) media;
            return (c.getCoverage().equals(this.coverage) && c.getFrequency() == this.frequency);
        }
        return false;
    }

    public boolean belongsToGroup(VlMediaGroup group) {
        if (!group.medias.isEmpty()) {
            return sameGroup(group.medias.get(0));
        }
        return false;
    }

    /**
     * Gets the channel hash code.
     * 
     * @return the hashcode
     */
    public int hashCode() {
        return frequency + coverage.hashCode() + name.hashCode();
    }

    public int getId() {
        return hashCode();
    }

    public boolean getIsRadio() {
        return category.contains("R-DIG");
    }

    public boolean getIsCrypted() {
        return category.contains("CRYPT");
    }

    public int compareTo(IVlMedia other) {
        if (other == null) {
            return -1;
        }
        if (other.getClass().equals(VlSatChannel.class)) {
            VlSatChannel channel = (VlSatChannel) other;
            if (channel.getFrequency() < frequency) {
                return 1;
            } else if (channel.getFrequency() > frequency) {
                return -1;
            } else {
                return name.compareTo(channel.getName());
            }
        }
        return -1;
    }
}
