/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.videolan.vlma.model.Adapter;
import org.videolan.vlma.model.FilesAdapter;
import org.videolan.vlma.model.FilesChannel;
import org.videolan.vlma.model.Media;
import org.videolan.vlma.model.MediaGroup;
import org.videolan.vlma.model.Order;
import org.videolan.vlma.model.Program;
import org.videolan.vlma.model.Server;
import org.videolan.vlma.order.OrderSender;

/**
 * This class computes the orders to give to the servers depending on the
 * medias programmation, on the state of servers state, and on the priority
 * of the programs.
 * <p>
 * To assign medias to servers, this class performs the following steps:
 * <ol>
 *   <li>A partition of available {@link Adapter}s is built according to their
 *   types.</li>
 *   <li>{@link Media}s are grouped together in {@link MediaGroup}s and
 *   partitioned according to the type of the {@link Adapter}s than can
 *   stream them.</li>
 *   <li>{@link MediaGroup}s are sorted by
 *   {@link Program#getPriority() priority} and {@link Order}s are given to
 *   {@link Adapter}s as long as there are available ones.</li>
 * </ol>
 * This means that the {@link MediaGroup}s with a low priority may not be
 * streamed.
 *
 * @author Adrien Maglo <magsoft at videolan.org>
 * @author Sylvain Cadilhac <sylv at videolan.org>
 * @author Adrien Grand <jpountz at videolan.org>
 */
public class OrderGiver {

    private static final Logger logger = Logger.getLogger(OrderGiver.class);

    // Time to sleep after orders have been sent
    public static final int SLEEP_AFTER_ORDERS = 5000;

    private VLMaService vlmaService;

    private OrderSender orderSender;

    private Thread computingThread;

    /**
     * Sets the VLMa service.
     *
     * @param vlmaService the vlmaService to set
     */
    public void setVlmaService(VLMaService vlmaService) {
        this.vlmaService = vlmaService;
    }

    /**
     * @param orderSender the orderSender to set
     */
    public void setOrderSender(OrderSender orderSender) {
        this.orderSender = orderSender;
    }

    /**
     * This method indicates whether or not the object is computing the orders.
     *
     * @return true only and only if the object is computing the orders. ordres
     */
    public boolean isComputing() {
        return (computingThread != null && computingThread.isAlive());
    }

    /**
     * Partition adapters according to their type.
     *
     * @param servers available servers
     * @return the partition of adapters
     */
    public Map<String, List<Adapter>> partitionAdapters(List<Server> servers) {
        int nbAdapters = 0;
        Map<String, List<Adapter>> adapters = new HashMap<String, List<Adapter>>();
        for(Server server : servers) {
            if (!server.isUp())
                continue;
            for(Adapter adapter : server.getAdapters()) {
                if (!adapter.isUp())
                    continue;
                nbAdapters += 1;
                List<Adapter> sameTypeAdapters = adapters.get(adapter.getType());
                if(sameTypeAdapters == null) {
                    sameTypeAdapters = new ArrayList<Adapter>();
                    adapters.put(adapter.getType(), sameTypeAdapters);
                }
                sameTypeAdapters.add(adapter);
            }
        }
        logger.debug(nbAdapters + " available adapters");
        return adapters;
    }

    /**
     * Partition medias according to the type of the adapters that can
     * stream them.
     *
     * @param medias medias to partition
     * @param adapters available adapters
     * @return the partition of media groups
     */
    public Map<String, List<MediaGroup>> partitionMedias(List<Media> medias, Map<String, List<Adapter>> adapters) {
        int nbMedias = 0;
        int nbGroups = 0;
        Map<String, List<MediaGroup>> groups = new HashMap<String, List<MediaGroup>>();
        for(Media media : medias) {
            if (media.getProgram() == null || !media.getProgram().isTimeToPlay())
                continue;
            nbMedias += 1;
            for(Entry<String, List<Adapter>> sameTypeAdapters : adapters.entrySet()) {
                if (!(sameTypeAdapters.getValue().get(0).canRead(media)
                        // Special case for files channels because of the possibility to set the server that must stream the channel
                        || (media instanceof FilesChannel &&  sameTypeAdapters.getValue().get(0) instanceof FilesAdapter)))
                    continue;
                List<MediaGroup> sameTypeGroups = groups.get(sameTypeAdapters.getKey());
                boolean mediaAdded = false;
                if (sameTypeGroups == null) {
                    sameTypeGroups = new ArrayList<MediaGroup>();
                    groups.put(sameTypeAdapters.getKey(), sameTypeGroups);
                } else {
                    for(MediaGroup group : sameTypeGroups) {
                        if (media.belongsToGroup(group)) {
                            group.add(media);
                            mediaAdded = true;
                            break;
                        }
                    }
                }
                if (!mediaAdded) {
                    MediaGroup group = new MediaGroup();
                    nbGroups += 1;
                    group.add(media);
                    sameTypeGroups.add(group);
                }
            }
        }
        return groups;
    }

    /**
     * Compute the list of orders that have to be sent in order the provided
     * mediagroup to be streamed using provided adapters.
     *
     * @param groups the media groups to be streamed
     * @param adapters available adapters
     * @return the set of corresponding orders
     */
    public Set<Order> computeOrders(Map<String, List<MediaGroup>> groups, Map<String, List<Adapter>> adapters) {
        Map<Server, Integer> nbMediasByServer = new HashMap<Server, Integer>();
        for(List<Adapter> adapterList : adapters.values()) {
            for(Adapter adapter : adapterList) {
                nbMediasByServer.put(adapter.getServer(), 0);
            }
        }
        Set<Order> orders = new HashSet<Order>();
        // First give orders to stream Satellite and DTT channels
        for (Map.Entry<String, List<MediaGroup>> entry : groups.entrySet()) {
            if (isSatOrDTT(entry.getKey()))
                computeOrdersByAvailableAdapters(entry.getValue(),
                        adapters.get(entry.getKey()),
                        orders,
                        nbMediasByServer);
        }
        // Then give other orders (for which there is not a limit of one
        // mediagroup per adapter) to servers with lower load
        for (Map.Entry<String, List<MediaGroup>> entry : groups.entrySet()) {
            if (!isSatOrDTT(entry.getKey()))
                computeOrdersByLowerLoad(entry.getValue(),
                        orders,
                        nbMediasByServer);
        }
        return orders;
    }

    private boolean isSatOrDTT(String key) {
        return key.startsWith(Adapter.DTT) || key.startsWith(Adapter.SAT);
    }

    private void computeOrdersByAvailableAdapters(List<MediaGroup> groups, List<Adapter> adapters, Set<Order> orders, Map<Server, Integer> nbMediasByServer) {
        // Handle groups with higher priority first
        Collections.sort(groups);
        Iterator<MediaGroup> groupIt = groups.iterator();
        Iterator<Adapter> adapterIt = adapters.iterator();
        while(groupIt.hasNext() && adapterIt.hasNext()) {
            Adapter a = adapterIt.next();
            MediaGroup g = groupIt.next();
            Order order = new Order(a, g);
            orders.add(order);
            int groupSize = g.size();
            nbMediasByServer.put(a.getServer(), nbMediasByServer.get(a.getServer()) + groupSize);
        }
        while(groupIt.hasNext()) {
            logger.warn("The following medias cannot be streamed because there is not enough adapters:");
            for(Media media : groupIt.next()) {
                logger.warn(" - " + media.getName());
            }
        }
    }

    private void computeOrdersByLowerLoad(List<MediaGroup> groups, Set<Order> orders, Map<Server, Integer> nbMediasByServer) {
        for(MediaGroup group : groups) {
            Adapter bestAdapter = null;
            int nbMediasBestAdapter = Integer.MAX_VALUE;
            for(Map.Entry<Server, Integer> entry : nbMediasByServer.entrySet()) {
                Server server = entry.getKey();
                int nbMedias = entry.getValue();
                for(Adapter adapter : server.getAdapters()) {
                    if(adapter.canRead(group) && nbMedias < nbMediasBestAdapter) {
                        bestAdapter = adapter;
                        nbMediasBestAdapter = nbMediasByServer.get(adapter.getServer());
                    }
                }
            }
            if(bestAdapter != null) {
                Order order = new Order(bestAdapter, group);
                orders.add(order);
                nbMediasByServer.put(bestAdapter.getServer(), nbMediasBestAdapter + group.size());
            } else {
                logger.warn("Cannot find any adapter for media group containing the following medias:");
                for(Media media : group) {
                    logger.warn(" - " + media.getName());
                }
            }
        }
    }

    /**
     * This method is the actual orders' computing method.
     */
    private Runnable orderComputer = new Runnable() {
        public void run() {
            logger.info("Starting computing channels assignment.");
            long start = System.currentTimeMillis();
            Map<String, List<Adapter>> adapters = partitionAdapters(vlmaService.getServers());
            Map<String, List<MediaGroup>> groups = partitionMedias(vlmaService.getMedias(), adapters);
            Set<Order> newOrders = computeOrders(groups, adapters);
            logger.info("Orders computation done in " + (System.currentTimeMillis() - start) + "ms");
            logger.info("Now sending orders");
            start = System.currentTimeMillis();
            if (logger.isDebugEnabled())
                logger.debug(newOrders.size() + " orders have been computed");
            Set<Order> oldOrders = new HashSet<Order>();
            Set<Order> orders = vlmaService.getOrders();
            synchronized(orders) {
                oldOrders.addAll(orders);

                // Remove former orders
                oldOrders.removeAll(newOrders);
                for (Order order : oldOrders) {
                    if (logger.isDebugEnabled())
                        logger.debug(oldOrders.size() + " orders to remove");
                    try {
                        orderSender.stop(order);
                    } catch (IOException e) {
                        logger.error("Error while trying to stop an order of " + order.getAdapter().getServer().getName() , e);
                        vlmaService.cancelOrder(order);
                    }
                    for(Media media : order.getMedias()) {
                        Program p = media.getProgram();
                        if(p != null) {
                            p.setPlayer(null);
                            p.setBroadcastState(false);
                        }
                    }
                }

                // Send newly computed orders in parallel
                newOrders.removeAll(orders);
                if (logger.isDebugEnabled())
                    logger.debug(newOrders.size() + " new orders to send");
                Map<Server, SendOrder> sendThreads = new HashMap<Server, SendOrder>();
                for (Order order : newOrders) {
                    for(Media media : order.getMedias()) {
                        media.getProgram().setPlayer(order.getAdapter().getServer().getIp());
                    }
                    SendOrder sendThread = sendThreads.get(order.getAdapter().getServer());
                    if(sendThread == null) {
                        sendThread = new SendOrder();
                        sendThreads.put(order.getAdapter().getServer(), sendThread);
                    }
                    sendThread.addOrder(order);
                }
                // Start threads, one per server
                for(Thread thread : sendThreads.values()) {
                    thread.start();
                }
                // Wait for all threads to finish
                for(Thread thread : sendThreads.values()) {
                    try {
                        thread.join();
                    } catch (InterruptedException e) { }
                }

                orders.removeAll(oldOrders);
                orders.addAll(newOrders);
                Set<Order> ordersToCancel = vlmaService.getOrdersToCancel();
                synchronized (ordersToCancel) {
                    ordersToCancel.removeAll(orders);
                }
            }
            logger.info("Orders sent in " + (System.currentTimeMillis() - start) + "ms");
            // Ensure that servers have enough time to effectively start
            // streaming orders that have been assigned to them before
            // OrderMonitor can check whether orders are streamed or not.
            try {
                Thread.sleep(SLEEP_AFTER_ORDERS);
            } catch (InterruptedException e) { }
        }
    };

    /**
     * This method launches the computing of orders in a new thread, if it was
     * not already running.
     */
    synchronized public void giveOrders() {
        if (!isComputing()) {
            computingThread = new Thread(orderComputer);
            computingThread.setName("OrderComputingThread");
            computingThread.start();
        }
    }

    private class SendOrder extends Thread {
        private List<Order> orders;

        public SendOrder() {
            orders = new ArrayList<Order>();
        }

        public void addOrder(Order order) {
            this.orders.add(order);
        }

        public void run() {
            for(Order order : orders) {
                try {
                    orderSender.start(order);
                } catch (IOException e) {
                    logger.error("Error while trying to send an order to " + order.getAdapter().getServer(), e);
                }
            }
        }
    }
}
